<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login atau bukan admin/panitia
if (!isLoggedIn() || !canUpload()) {
    header("Location: dashboard.php");
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    $user_id = $_SESSION['user_id'];
    $sql = "SELECT full_name FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $_SESSION['full_name'] = $user['full_name'];
    }
}

$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] == 'add') {
            $jemaat_id = (int)$_POST['jemaat_id'];
            $nama = trim($_POST['nama']);
            $tanggal_lahir = $_POST['tanggal_lahir'];
            $alamat = trim($_POST['alamat']);
            $nomor_hp = trim($_POST['nomor_hp']);
            $jenis_kelamin = $_POST['jenis_kelamin'];
            
            if (empty($nama) || empty($tanggal_lahir) || empty($jenis_kelamin)) {
                $error = 'Nama, tanggal lahir, dan jenis kelamin harus diisi!';
            } else {
                // Handle foto upload
                $foto_peserta = '';
                if (isset($_FILES['foto_peserta']) && $_FILES['foto_peserta']['error'] === UPLOAD_ERR_OK) {
                    $file = $_FILES['foto_peserta'];
                    $original_filename = $file['name'];
                    $file_size = $file['size'];
                    $tmp_name = $file['tmp_name'];
                    
                    // Validasi tipe file
                    if (!isImage($original_filename)) {
                        $error = 'File foto harus berupa gambar (JPG, JPEG, PNG, GIF)!';
                    } elseif ($file_size > 5 * 1024 * 1024) { // 5MB limit
                        $error = 'Ukuran foto terlalu besar! Maksimal 5MB';
                    } else {
                        // Buat direktori uploads jika belum ada
                        $upload_dir = 'uploads/peserta/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0777, true);
                        }
                        
                        // Generate nama file unik
                        $filename = generateUniqueFilename($original_filename);
                        $file_path = $upload_dir . $filename;
                        
                        // Upload file
                        if (move_uploaded_file($tmp_name, $file_path)) {
                            $foto_peserta = $filename;
                        } else {
                            $error = 'Gagal mengupload foto!';
                        }
                    }
                }
                
                if (empty($error)) {
                    $sql = "INSERT INTO peserta (jemaat_id, nama, tanggal_lahir, alamat, nomor_hp, jenis_kelamin, foto_peserta) VALUES (?, ?, ?, ?, ?, ?, ?)";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("issssss", $jemaat_id, $nama, $tanggal_lahir, $alamat, $nomor_hp, $jenis_kelamin, $foto_peserta);
                    
                    if ($stmt->execute()) {
                        $peserta_id = $conn->insert_id;
                        $success = 'Data peserta berhasil ditambahkan!';
                        
                        // Log aktivitas CREATE
                        logActivity($conn, 'CREATE', 'peserta', $peserta_id, "Menambahkan peserta: $nama");
                        
                        // Reset form
                        $nama = $alamat = $nomor_hp = '';
                        $tanggal_lahir = '';
                        $jenis_kelamin = '';
                    } else {
                        $error = 'Gagal menambahkan data peserta!';
                        // Hapus foto jika gagal
                        if (!empty($foto_peserta)) {
                            unlink($upload_dir . $foto_peserta);
                        }
                    }
                }
            }
        } elseif ($_POST['action'] == 'edit') {
            $peserta_id = (int)$_POST['peserta_id'];
            $jemaat_id = (int)$_POST['jemaat_id'];
            $nama = trim($_POST['nama']);
            $tanggal_lahir = $_POST['tanggal_lahir'];
            $alamat = trim($_POST['alamat']);
            $nomor_hp = trim($_POST['nomor_hp']);
            $jenis_kelamin = $_POST['jenis_kelamin'];
            
            if (empty($nama) || empty($tanggal_lahir) || empty($jenis_kelamin)) {
                $error = 'Nama, tanggal lahir, dan jenis kelamin harus diisi!';
            } else {
                // Handle foto upload
                $foto_peserta = $_POST['foto_peserta_old'];
                if (isset($_FILES['foto_peserta']) && $_FILES['foto_peserta']['error'] === UPLOAD_ERR_OK) {
                    $file = $_FILES['foto_peserta'];
                    $original_filename = $file['name'];
                    $file_size = $file['size'];
                    $tmp_name = $file['tmp_name'];
                    
                    // Validasi tipe file
                    if (!isImage($original_filename)) {
                        $error = 'File foto harus berupa gambar (JPG, JPEG, PNG, GIF)!';
                    } elseif ($file_size > 5 * 1024 * 1024) { // 5MB limit
                        $error = 'Ukuran foto terlalu besar! Maksimal 5MB';
                    } else {
                        // Buat direktori uploads jika belum ada
                        $upload_dir = 'uploads/peserta/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0777, true);
                        }
                        
                        // Generate nama file unik
                        $filename = generateUniqueFilename($original_filename);
                        $file_path = $upload_dir . $filename;
                        
                        // Upload file
                        if (move_uploaded_file($tmp_name, $file_path)) {
                            // Hapus foto lama jika ada
                            if (!empty($_POST['foto_peserta_old'])) {
                                $old_file = $upload_dir . $_POST['foto_peserta_old'];
                                if (file_exists($old_file)) {
                                    unlink($old_file);
                                }
                            }
                            $foto_peserta = $filename;
                        } else {
                            $error = 'Gagal mengupload foto!';
                        }
                    }
                }
                
                if (empty($error)) {
                    $sql = "UPDATE peserta SET jemaat_id = ?, nama = ?, tanggal_lahir = ?, alamat = ?, nomor_hp = ?, jenis_kelamin = ?, foto_peserta = ? WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("issssssi", $jemaat_id, $nama, $tanggal_lahir, $alamat, $nomor_hp, $jenis_kelamin, $foto_peserta, $peserta_id);
                    
                    if ($stmt->execute()) {
                        $success = 'Data peserta berhasil diupdate!';
                        
                        // Log aktivitas UPDATE
                        logActivity($conn, 'UPDATE', 'peserta', $peserta_id, "Mengupdate peserta: $nama");
                    } else {
                        $error = 'Gagal mengupdate data peserta!';
                        // Hapus foto baru jika gagal
                        if ($foto_peserta != $_POST['foto_peserta_old']) {
                            unlink($upload_dir . $foto_peserta);
                        }
                    }
                }
            }
        } elseif ($_POST['action'] == 'import') {
            // Import data peserta dari Excel/CSV
            if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
                $error = 'Pilih file untuk diimport!';
            } else {
                $file = $_FILES['import_file'];
                $file_size = $file['size'];
                $file_tmp = $file['tmp_name'];
                $file_name = $file['name'];
                $file_ext = strtolower(pathinfo($file_name, PATHINFO_EXTENSION));
                
                // Validasi file
                if ($file_size > 5 * 1024 * 1024) { // 5MB limit
                    $error = 'Ukuran file terlalu besar! Maksimal 5MB';
                } elseif (!in_array($file_ext, ['xlsx', 'xls', 'csv'])) {
                    $error = 'Format file tidak didukung! Gunakan Excel (.xlsx, .xls) atau CSV (.csv)';
                } else {
                    // Proses import
                    $import_jemaat_id = (int)$_POST['import_jemaat_id'];
                    $skip_duplicate = isset($_POST['skip_duplicate']);
                    
                    if ($import_jemaat_id <= 0) {
                        $error = 'Pilih jemaat default!';
                    } else {
                        // Cek apakah jemaat ada
                        $check_jemaat = $conn->prepare("SELECT id FROM jemaat WHERE id = ?");
                        $check_jemaat->bind_param("i", $import_jemaat_id);
                        $check_jemaat->execute();
                        if ($check_jemaat->get_result()->num_rows == 0) {
                            $error = 'Jemaat yang dipilih tidak ditemukan!';
                        } else {
                            // Proses file
                            $import_result = processImportFile($conn, $file_tmp, $file_ext, $import_jemaat_id, $skip_duplicate);
                            
                            if ($import_result['success']) {
                                $success = $import_result['message'];
                                
                                // Log aktivitas IMPORT
                                logActivity($conn, 'IMPORT', 'peserta', null, "Import data peserta: " . $import_result['success_count'] . " berhasil, " . $import_result['error_count'] . " gagal");
                            } else {
                                $error = $import_result['message'];
                            }
                        }
                    }
                }
            }
        } elseif ($_POST['action'] == 'delete') {
            $peserta_id = (int)$_POST['peserta_id'];
            
            // Ambil nama file foto sebelum hapus
            $sql = "SELECT foto_peserta FROM peserta WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $peserta_id);
            $stmt->execute();
            $result = $stmt->get_result();
            $peserta = $result->fetch_assoc();
            
            // Hapus data dari database
            $sql = "DELETE FROM peserta WHERE id = ?";
            $stmt = $conn->prepare($sql);
            $stmt->bind_param("i", $peserta_id);
            
            if ($stmt->execute()) {
                // Hapus file foto jika ada
                if (!empty($peserta['foto_peserta'])) {
                    $file_path = 'uploads/peserta/' . $peserta['foto_peserta'];
                    if (file_exists($file_path)) {
                        unlink($file_path);
                    }
                }
                $success = 'Data peserta berhasil dihapus!';
                
                // Log aktivitas DELETE
                logActivity($conn, 'DELETE', 'peserta', $peserta_id, "Menghapus peserta: " . $peserta['nama']);
            } else {
                $error = 'Gagal menghapus data peserta!';
            }
        }
    }
}

// Ambil data jemaat untuk dropdown
$jemaat_sql = "SELECT id, nama_jemaat FROM jemaat ORDER BY nama_jemaat";
$jemaat_result = $conn->query($jemaat_sql);

// Filter jemaat
$jemaat_filter = isset($_GET['jemaat']) ? (int)$_GET['jemaat'] : 0;

// Ambil data peserta dengan join jemaat
$sql = "SELECT p.*, j.nama_jemaat FROM peserta p 
        LEFT JOIN jemaat j ON p.jemaat_id = j.id";

$where_clause = "";
$params = [];
$types = "";

if ($jemaat_filter > 0) {
    $where_clause = "WHERE p.jemaat_id = ?";
    $params[] = $jemaat_filter;
    $types = "i";
}

if (!empty($where_clause)) {
    $sql .= " " . $where_clause;
}

$sql .= " ORDER BY p.created_at DESC";

if (!empty($params)) {
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
} else {
    $result = $conn->query($sql);
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Data Peserta - Sistem Manajemen Dokumen</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Sistem Dokumen Persidangan
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-home me-1"></i>Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="upload.php">
                            <i class="fas fa-upload me-1"></i>Upload Dokumen
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jemaat.php">
                            <i class="fas fa-church me-1"></i>Data Jemaat
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="peserta.php">
                            <i class="fas fa-users me-1"></i>Data Peserta
                        </a>
                    </li>
                    <?php if (hasRole('administrator') || hasRole('panitia')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="activity_logs.php">
                            <i class="fas fa-history me-1"></i>Log Aktivitas
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="profile.php">
                                <i class="fas fa-user-edit me-2"></i>Profile
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php">
                                <i class="fas fa-sign-out-alt me-2"></i>Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Header -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h2><i class="fas fa-users me-2"></i>Data Peserta</h2>
                <p class="text-muted">Kelola data peserta persidangan</p>
            </div>
            <div class="col-md-4 text-end">
                <button class="btn btn-success me-2" data-bs-toggle="modal" data-bs-target="#importPesertaModal">
                    <i class="fas fa-file-import me-2"></i>Import Data
                </button>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addPesertaModal">
                    <i class="fas fa-plus me-2"></i>Tambah Peserta
                </button>
            </div>
        </div>

        <!-- Alerts -->
        <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Filter -->
        <div class="card mb-4">
            <div class="card-body">
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <label for="jemaat" class="form-label">Filter Jemaat</label>
                        <select class="form-select" id="jemaat" name="jemaat">
                            <option value="0">Semua Jemaat</option>
                            <?php while ($jemaat = $jemaat_result->fetch_assoc()): ?>
                            <option value="<?php echo $jemaat['id']; ?>" <?php echo $jemaat_filter == $jemaat['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($jemaat['nama_jemaat']); ?>
                            </option>
                            <?php endwhile; ?>
                        </select>
                    </div>
                    <div class="col-md-4 d-flex align-items-end">
                        <button type="submit" class="btn btn-outline-primary me-2">
                            <i class="fas fa-filter me-1"></i>Filter
                        </button>
                        <a href="peserta.php" class="btn btn-outline-secondary">
                            <i class="fas fa-times me-1"></i>Reset
                        </a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Peserta Table -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>ID</th>
                                <th>Foto</th>
                                <th>Nama</th>
                                <th>Tanggal Lahir</th>
                                <th>Jenis Kelamin</th>
                                <th>Alamat</th>
                                <th>No. HP</th>
                                <th>Jemaat</th>
                                <th>Status</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php while ($peserta = $result->fetch_assoc()): ?>
                            <tr>
                                <td><?php echo $peserta['id']; ?></td>
                                <td>
                                    <?php if (!empty($peserta['foto_peserta'])): ?>
                                        <img src="uploads/peserta/<?php echo $peserta['foto_peserta']; ?>" 
                                             alt="Foto <?php echo htmlspecialchars($peserta['nama']); ?>" 
                                             class="rounded-circle" width="40" height="40" style="object-fit: cover;">
                                    <?php else: ?>
                                        <div class="bg-secondary rounded-circle d-flex align-items-center justify-content-center" 
                                             style="width: 40px; height: 40px;">
                                            <i class="fas fa-user text-white"></i>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td><strong><?php echo htmlspecialchars($peserta['nama']); ?></strong></td>
                                <td><?php echo date('d/m/Y', strtotime($peserta['tanggal_lahir'])); ?></td>
                                <td>
                                    <span class="badge bg-<?php echo $peserta['jenis_kelamin'] == 'Laki-laki' ? 'primary' : 'pink'; ?>">
                                        <?php echo $peserta['jenis_kelamin']; ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($peserta['alamat']); ?></td>
                                <td><?php echo htmlspecialchars($peserta['nomor_hp']); ?></td>
                                <td>
                                    <span class="badge bg-info">
                                        <?php echo htmlspecialchars($peserta['nama_jemaat']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?php echo $peserta['status'] == 'Aktif' ? 'success' : 'secondary'; ?>">
                                        <?php echo $peserta['status']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="btn-group btn-group-sm">
                                        <button class="btn btn-outline-primary" onclick="editPeserta(<?php echo htmlspecialchars(json_encode($peserta)); ?>)" title="Edit">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="btn btn-outline-danger" onclick="deletePeserta(<?php echo $peserta['id']; ?>, '<?php echo htmlspecialchars($peserta['nama']); ?>')" title="Hapus">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endwhile; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Peserta Modal -->
    <div class="modal fade" id="addPesertaModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-user-plus me-2"></i>Tambah Peserta Baru
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="add">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="jemaat_id" class="form-label">Jemaat <span class="text-danger">*</span></label>
                                    <select class="form-select" id="jemaat_id" name="jemaat_id" required>
                                        <option value="">Pilih Jemaat</option>
                                        <?php 
                                        $jemaat_result->data_seek(0);
                                        while ($jemaat = $jemaat_result->fetch_assoc()): 
                                        ?>
                                        <option value="<?php echo $jemaat['id']; ?>">
                                            <?php echo htmlspecialchars($jemaat['nama_jemaat']); ?>
                                        </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="nama" class="form-label">Nama Lengkap <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="nama" name="nama" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="tanggal_lahir" class="form-label">Tanggal Lahir <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="tanggal_lahir" name="tanggal_lahir" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="jenis_kelamin" class="form-label">Jenis Kelamin <span class="text-danger">*</span></label>
                                    <select class="form-select" id="jenis_kelamin" name="jenis_kelamin" required>
                                        <option value="">Pilih Jenis Kelamin</option>
                                        <option value="Laki-laki">Laki-laki</option>
                                        <option value="Perempuan">Perempuan</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="alamat" class="form-label">Alamat</label>
                                    <textarea class="form-control" id="alamat" name="alamat" rows="3"></textarea>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="nomor_hp" class="form-label">Nomor HP</label>
                                    <input type="tel" class="form-control" id="nomor_hp" name="nomor_hp">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="foto_peserta" class="form-label">Foto Peserta</label>
                                    <input type="file" class="form-control" id="foto_peserta" name="foto_peserta" accept="image/*">
                                    <small class="text-muted">Format: JPG, JPEG, PNG, GIF. Maksimal 5MB</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Preview Foto</label>
                                    <div id="foto_preview" class="border rounded p-2 text-center" style="min-height: 100px;">
                                        <i class="fas fa-image text-muted" style="font-size: 2rem;"></i>
                                        <p class="text-muted mt-2">Foto akan ditampilkan di sini</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Peserta Modal -->
    <div class="modal fade" id="editPesertaModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-user-edit me-2"></i>Edit Peserta
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="edit">
                        <input type="hidden" name="peserta_id" id="edit_peserta_id">
                        <input type="hidden" name="foto_peserta_old" id="edit_foto_peserta_old">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_jemaat_id" class="form-label">Jemaat <span class="text-danger">*</span></label>
                                    <select class="form-select" id="edit_jemaat_id" name="jemaat_id" required>
                                        <option value="">Pilih Jemaat</option>
                                        <?php 
                                        $jemaat_result->data_seek(0);
                                        while ($jemaat = $jemaat_result->fetch_assoc()): 
                                        ?>
                                        <option value="<?php echo $jemaat['id']; ?>">
                                            <?php echo htmlspecialchars($jemaat['nama_jemaat']); ?>
                                        </option>
                                        <?php endwhile; ?>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="edit_nama" class="form-label">Nama Lengkap <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="edit_nama" name="nama" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="edit_tanggal_lahir" class="form-label">Tanggal Lahir <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="edit_tanggal_lahir" name="tanggal_lahir" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="edit_jenis_kelamin" class="form-label">Jenis Kelamin <span class="text-danger">*</span></label>
                                    <select class="form-select" id="edit_jenis_kelamin" name="jenis_kelamin" required>
                                        <option value="">Pilih Jenis Kelamin</option>
                                        <option value="Laki-laki">Laki-laki</option>
                                        <option value="Perempuan">Perempuan</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_alamat" class="form-label">Alamat</label>
                                    <textarea class="form-control" id="edit_alamat" name="alamat" rows="3"></textarea>
                                </div>
                                
                                <div class="mb-3">
                                    <label for="edit_nomor_hp" class="form-label">Nomor HP</label>
                                    <input type="tel" class="form-control" id="edit_nomor_hp" name="nomor_hp">
                                </div>
                                
                                <div class="mb-3">
                                    <label for="edit_foto_peserta" class="form-label">Foto Peserta</label>
                                    <input type="file" class="form-control" id="edit_foto_peserta" name="foto_peserta" accept="image/*">
                                    <small class="text-muted">Format: JPG, JPEG, PNG, GIF. Maksimal 5MB</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Preview Foto</label>
                                    <div id="edit_foto_preview" class="border rounded p-2 text-center" style="min-height: 100px;">
                                        <i class="fas fa-image text-muted" style="font-size: 2rem;"></i>
                                        <p class="text-muted mt-2">Foto akan ditampilkan di sini</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Import Peserta Modal -->
    <div class="modal fade" id="importPesertaModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-file-import me-2"></i>Import Data Peserta
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" enctype="multipart/form-data" id="importForm">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="import">
                        
                        <div class="alert alert-info">
                            <h6><i class="fas fa-info-circle me-2"></i>Petunjuk Import Data:</h6>
                            <ul class="mb-0">
                                <li>File harus berformat Excel (.xlsx) atau CSV (.csv)</li>
                                <li>Kolom wajib: Nama, Tanggal Lahir, Jenis Kelamin, Jemaat</li>
                                <li>Kolom opsional: Alamat, Nomor HP</li>
                                <li>Format tanggal: YYYY-MM-DD (contoh: 1990-05-15)</li>
                                <li>Jenis Kelamin: Laki-laki atau Perempuan</li>
                                <li>Jemaat harus sudah ada di sistem</li>
                            </ul>
                        </div>
                        
                        <div class="mb-3">
                            <label for="import_file" class="form-label">File Excel/CSV <span class="text-danger">*</span></label>
                            <input type="file" class="form-control" id="import_file" name="import_file" accept=".xlsx,.xls,.csv" required>
                            <small class="text-muted">Maksimal 5MB</small>
                        </div>
                        
                        <div class="mb-3">
                            <label for="import_jemaat_id" class="form-label">Jemaat Default <span class="text-danger">*</span></label>
                            <select class="form-select" id="import_jemaat_id" name="import_jemaat_id" required>
                                <option value="">Pilih Jemaat</option>
                                <?php 
                                $jemaat_result->data_seek(0);
                                while ($jemaat = $jemaat_result->fetch_assoc()): 
                                ?>
                                <option value="<?php echo $jemaat['id']; ?>">
                                    <?php echo htmlspecialchars($jemaat['nama_jemaat']); ?>
                                </option>
                                <?php endwhile; ?>
                            </select>
                            <small class="text-muted">Jemaat default jika tidak ada kolom jemaat di file</small>
                        </div>
                        
                        <div class="mb-3">
                            <div class="form-check">
                                <input class="form-check-input" type="checkbox" id="skip_duplicate" name="skip_duplicate" checked>
                                <label class="form-check-label" for="skip_duplicate">
                                    Skip data duplikat (berdasarkan nama dan tanggal lahir)
                                </label>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <a href="template_import_peserta.xlsx" class="btn btn-outline-primary btn-sm">
                                <i class="fas fa-download me-1"></i>Download Template Excel
                            </a>
                            <a href="template_import_peserta.csv" class="btn btn-outline-secondary btn-sm">
                                <i class="fas fa-download me-1"></i>Download Template CSV
                            </a>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-upload me-2"></i>Import Data
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deletePesertaModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle text-warning me-2"></i>Konfirmasi Hapus
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Apakah Anda yakin ingin menghapus peserta <strong id="delete_nama"></strong>?</p>
                    <p class="text-danger">Tindakan ini tidak dapat dibatalkan!</p>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="peserta_id" id="delete_peserta_id">
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="fas fa-trash me-2"></i>Hapus
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Preview foto untuk form tambah
        document.getElementById('foto_peserta').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('foto_preview');
            
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.innerHTML = `<img src="${e.target.result}" class="img-fluid rounded" style="max-height: 150px;">`;
                };
                reader.readAsDataURL(file);
            } else {
                preview.innerHTML = `
                    <i class="fas fa-image text-muted" style="font-size: 2rem;"></i>
                    <p class="text-muted mt-2">Foto akan ditampilkan di sini</p>
                `;
            }
        });
        
        // Preview foto untuk form edit
        document.getElementById('edit_foto_peserta').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('edit_foto_preview');
            
            if (file) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.innerHTML = `<img src="${e.target.result}" class="img-fluid rounded" style="max-height: 150px;">`;
                };
                reader.readAsDataURL(file);
            } else {
                // Tampilkan foto lama jika ada
                const oldFoto = document.getElementById('edit_foto_peserta_old').value;
                if (oldFoto) {
                    preview.innerHTML = `<img src="uploads/peserta/${oldFoto}" class="img-fluid rounded" style="max-height: 150px;">`;
                } else {
                    preview.innerHTML = `
                        <i class="fas fa-image text-muted" style="font-size: 2rem;"></i>
                        <p class="text-muted mt-2">Foto akan ditampilkan di sini</p>
                    `;
                }
            }
        });
        
        function editPeserta(peserta) {
            document.getElementById('edit_peserta_id').value = peserta.id;
            document.getElementById('edit_jemaat_id').value = peserta.jemaat_id;
            document.getElementById('edit_nama').value = peserta.nama;
            document.getElementById('edit_tanggal_lahir').value = peserta.tanggal_lahir;
            document.getElementById('edit_alamat').value = peserta.alamat;
            document.getElementById('edit_nomor_hp').value = peserta.nomor_hp;
            document.getElementById('edit_jenis_kelamin').value = peserta.jenis_kelamin;
            document.getElementById('edit_foto_peserta_old').value = peserta.foto_peserta;
            
            // Reset file input
            document.getElementById('edit_foto_peserta').value = '';
            
            // Tampilkan foto lama
            const preview = document.getElementById('edit_foto_preview');
            if (peserta.foto_peserta) {
                preview.innerHTML = `<img src="uploads/peserta/${peserta.foto_peserta}" class="img-fluid rounded" style="max-height: 150px;">`;
            } else {
                preview.innerHTML = `
                    <i class="fas fa-image text-muted" style="font-size: 2rem;"></i>
                    <p class="text-muted mt-2">Tidak ada foto</p>
                `;
            }
            
            new bootstrap.Modal(document.getElementById('editPesertaModal')).show();
        }
        
        function deletePeserta(pesertaId, nama) {
            document.getElementById('delete_peserta_id').value = pesertaId;
            document.getElementById('delete_nama').textContent = nama;
            
            new bootstrap.Modal(document.getElementById('deletePesertaModal')).show();
        }
    </script>
</body>
</html> 