<?php
// Fungsi untuk mengecek apakah user sudah login
function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

// Fungsi untuk mengecek role user
function hasRole($role) {
    return isset($_SESSION['role']) && $_SESSION['role'] === $role;
}

// Fungsi untuk mengecek apakah user bisa upload (admin atau panitia)
function canUpload() {
    return hasRole('administrator') || hasRole('panitia');
}

// Fungsi untuk memproses import file Excel/CSV
function processImportFile($conn, $file_path, $file_ext, $default_jemaat_id, $skip_duplicate = true) {
    $success_count = 0;
    $error_count = 0;
    $errors = [];
    
    try {
        if ($file_ext == 'csv') {
            // Proses file CSV
            $handle = fopen($file_path, 'r');
            if ($handle === false) {
                return ['success' => false, 'message' => 'Gagal membuka file CSV'];
            }
            
            // Baca header
            $headers = fgetcsv($handle);
            if (!$headers) {
                fclose($handle);
                return ['success' => false, 'message' => 'File CSV kosong atau format tidak valid'];
            }
            
            // Mapping kolom
            $column_map = [];
            foreach ($headers as $index => $header) {
                $header = trim(strtolower($header));
                switch ($header) {
                    case 'nama':
                    case 'name':
                        $column_map['nama'] = $index;
                        break;
                    case 'tanggal_lahir':
                    case 'tanggal lahir':
                    case 'birth_date':
                    case 'birthdate':
                        $column_map['tanggal_lahir'] = $index;
                        break;
                    case 'jenis_kelamin':
                    case 'jenis kelamin':
                    case 'gender':
                        $column_map['jenis_kelamin'] = $index;
                        break;
                    case 'alamat':
                    case 'address':
                        $column_map['alamat'] = $index;
                        break;
                    case 'nomor_hp':
                    case 'nomor hp':
                    case 'phone':
                    case 'telepon':
                        $column_map['nomor_hp'] = $index;
                        break;
                    case 'jemaat':
                    case 'jemaat_id':
                        $column_map['jemaat'] = $index;
                        break;
                }
            }
            
            // Validasi kolom wajib
            if (!isset($column_map['nama']) || !isset($column_map['tanggal_lahir']) || !isset($column_map['jenis_kelamin'])) {
                fclose($handle);
                return ['success' => false, 'message' => 'Kolom wajib tidak ditemukan: Nama, Tanggal Lahir, Jenis Kelamin'];
            }
            
            // Proses data
            $row_number = 1; // Mulai dari 1 karena header
            while (($data = fgetcsv($handle)) !== false) {
                $row_number++;
                
                $result = processImportRow($conn, $data, $column_map, $default_jemaat_id, $skip_duplicate, $row_number);
                if ($result['success']) {
                    $success_count++;
                } else {
                    $error_count++;
                    $errors[] = "Baris $row_number: " . $result['message'];
                }
            }
            
            fclose($handle);
            
        } else {
            // Proses file Excel (xlsx, xls)
            // Untuk sementara, kita akan menggunakan library PhpSpreadsheet
            // Tapi karena tidak tersedia, kita akan return error
            return ['success' => false, 'message' => 'Import Excel belum didukung. Gunakan format CSV.'];
        }
        
        // Buat pesan hasil
        $message = "Import selesai. Berhasil: $success_count, Gagal: $error_count";
        if (!empty($errors)) {
            $message .= "\nError detail:\n" . implode("\n", array_slice($errors, 0, 10)); // Tampilkan max 10 error
            if (count($errors) > 10) {
                $message .= "\n... dan " . (count($errors) - 10) . " error lainnya";
            }
        }
        
        return [
            'success' => $success_count > 0,
            'message' => $message,
            'success_count' => $success_count,
            'error_count' => $error_count
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
    }
}

// Fungsi untuk memproses satu baris data import
function processImportRow($conn, $data, $column_map, $default_jemaat_id, $skip_duplicate, $row_number) {
    try {
        // Ambil data dari kolom
        $nama = trim($data[$column_map['nama']] ?? '');
        $tanggal_lahir = trim($data[$column_map['tanggal_lahir']] ?? '');
        $jenis_kelamin = trim($data[$column_map['jenis_kelamin']] ?? '');
        $alamat = trim($data[$column_map['alamat']] ?? '');
        $nomor_hp = trim($data[$column_map['nomor_hp']] ?? '');
        $jemaat_id = $default_jemaat_id;
        
        // Jika ada kolom jemaat, gunakan itu
        if (isset($column_map['jemaat'])) {
            $jemaat_value = trim($data[$column_map['jemaat']] ?? '');
            if (!empty($jemaat_value)) {
                // Cari jemaat berdasarkan nama
                $stmt = $conn->prepare("SELECT id FROM jemaat WHERE nama_jemaat = ?");
                $stmt->bind_param("s", $jemaat_value);
                $stmt->execute();
                $result = $stmt->get_result();
                if ($result->num_rows > 0) {
                    $jemaat = $result->fetch_assoc();
                    $jemaat_id = $jemaat['id'];
                } else {
                    return ['success' => false, 'message' => "Jemaat '$jemaat_value' tidak ditemukan"];
                }
            }
        }
        
        // Validasi data
        if (empty($nama)) {
            return ['success' => false, 'message' => 'Nama tidak boleh kosong'];
        }
        
        if (empty($tanggal_lahir)) {
            return ['success' => false, 'message' => 'Tanggal lahir tidak boleh kosong'];
        }
        
        // Validasi format tanggal
        $date_obj = DateTime::createFromFormat('Y-m-d', $tanggal_lahir);
        if (!$date_obj || $date_obj->format('Y-m-d') !== $tanggal_lahir) {
            return ['success' => false, 'message' => 'Format tanggal lahir tidak valid (gunakan YYYY-MM-DD)'];
        }
        
        if (empty($jenis_kelamin)) {
            return ['success' => false, 'message' => 'Jenis kelamin tidak boleh kosong'];
        }
        
        // Validasi jenis kelamin
        $jenis_kelamin = ucfirst(strtolower($jenis_kelamin));
        if (!in_array($jenis_kelamin, ['Laki-laki', 'Perempuan'])) {
            return ['success' => false, 'message' => 'Jenis kelamin harus Laki-laki atau Perempuan'];
        }
        
        // Cek duplikat jika skip_duplicate = true
        if ($skip_duplicate) {
            $stmt = $conn->prepare("SELECT id FROM peserta WHERE nama = ? AND tanggal_lahir = ?");
            $stmt->bind_param("ss", $nama, $tanggal_lahir);
            $stmt->execute();
            if ($stmt->get_result()->num_rows > 0) {
                return ['success' => false, 'message' => 'Data duplikat (nama dan tanggal lahir sama)'];
            }
        }
        
        // Insert ke database
        $stmt = $conn->prepare("INSERT INTO peserta (jemaat_id, nama, tanggal_lahir, alamat, nomor_hp, jenis_kelamin) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("isssss", $jemaat_id, $nama, $tanggal_lahir, $alamat, $nomor_hp, $jenis_kelamin);
        
        if ($stmt->execute()) {
            return ['success' => true, 'message' => 'Berhasil'];
        } else {
            return ['success' => false, 'message' => 'Gagal menyimpan ke database: ' . $stmt->error];
        }
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
    }
}

// Fungsi untuk format ukuran file
function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

// Fungsi untuk mendapatkan ekstensi file
function getFileExtension($filename) {
    return strtolower(pathinfo($filename, PATHINFO_EXTENSION));
}

// Fungsi untuk mengecek tipe file yang diizinkan
function isAllowedFileType($filename) {
    $allowed_types = ['pdf', 'jpg', 'jpeg', 'png', 'gif', 'mp4', 'avi', 'mov', 'wmv'];
    $extension = getFileExtension($filename);
    return in_array($extension, $allowed_types);
}

// Fungsi untuk menentukan kategori file
function getFileCategory($filename) {
    $extension = getFileExtension($filename);
    if ($extension === 'pdf') {
        return 'pdf';
    } elseif (in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
        return 'image';
    } elseif (in_array($extension, ['mp4', 'avi', 'mov', 'wmv'])) {
        return 'video';
    }
    return 'unknown';
}

// Fungsi untuk generate nama file unik
function generateUniqueFilename($original_filename) {
    $extension = getFileExtension($original_filename);
    return uniqid() . '_' . time() . '.' . $extension;
}

// Fungsi untuk mengecek apakah file adalah gambar
function isImage($filename) {
    $extension = getFileExtension($filename);
    return in_array($extension, ['jpg', 'jpeg', 'png', 'gif']);
}

// Fungsi untuk mengecek apakah file adalah video
function isVideo($filename) {
    $extension = getFileExtension($filename);
    return in_array($extension, ['mp4', 'avi', 'mov', 'wmv']);
}

// Fungsi untuk mengecek apakah file adalah PDF
function isPDF($filename) {
    return getFileExtension($filename) === 'pdf';
}

// Fungsi untuk mendapatkan icon berdasarkan tipe file
function getFileIcon($file_type) {
    switch ($file_type) {
        case 'pdf':
            return 'fas fa-file-pdf text-danger';
        case 'image':
            return 'fas fa-file-image text-success';
        case 'video':
            return 'fas fa-file-video text-primary';
        default:
            return 'fas fa-file text-secondary';
    }
}

// Fungsi untuk mendapatkan warna badge berdasarkan tipe file
function getFileBadgeColor($file_type) {
    switch ($file_type) {
        case 'pdf':
            return 'danger';
        case 'image':
            return 'success';
        case 'video':
            return 'primary';
        default:
            return 'secondary';
    }
}

// ==================== FUNGSI LOG AKTIVITAS ====================

// Fungsi untuk mencatat log aktivitas
function logActivity($conn, $action, $table_name = null, $record_id = null, $description = null) {
    if (!isset($_SESSION['user_id'])) {
        return false; // Tidak bisa log jika user tidak login
    }
    
    $user_id = $_SESSION['user_id'];
    $username = $_SESSION['username'] ?? 'Unknown';
    $full_name = $_SESSION['full_name'] ?? 'Unknown';
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    $stmt = $conn->prepare("INSERT INTO activity_logs (user_id, username, full_name, action, table_name, record_id, description, ip_address, user_agent) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("issssisss", $user_id, $username, $full_name, $action, $table_name, $record_id, $description, $ip_address, $user_agent);
    
    return $stmt->execute();
}

// Fungsi untuk mencatat login
function logLogin($conn, $user_id, $username, $full_name) {
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    $stmt = $conn->prepare("INSERT INTO activity_logs (user_id, username, full_name, action, ip_address, user_agent) VALUES (?, ?, ?, 'LOGIN', ?, ?)");
    $stmt->bind_param("issss", $user_id, $username, $full_name, $ip_address, $user_agent);
    
    return $stmt->execute();
}

// Fungsi untuk mencatat logout
function logLogout($conn) {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    $user_id = $_SESSION['user_id'];
    $username = $_SESSION['username'] ?? 'Unknown';
    $full_name = $_SESSION['full_name'] ?? 'Unknown';
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'Unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
    
    $stmt = $conn->prepare("INSERT INTO activity_logs (user_id, username, full_name, action, ip_address, user_agent) VALUES (?, ?, ?, 'LOGOUT', ?, ?)");
    $stmt->bind_param("issss", $user_id, $username, $full_name, $ip_address, $user_agent);
    
    return $stmt->execute();
}

// Fungsi untuk mendapatkan log aktivitas dengan filter
function getActivityLogs($conn, $limit = 100, $offset = 0, $user_id = null, $action = null, $table_name = null, $date_from = null, $date_to = null) {
    $where_conditions = [];
    $params = [];
    $types = "";
    
    if ($user_id) {
        $where_conditions[] = "al.user_id = ?";
        $params[] = $user_id;
        $types .= "i";
    }
    
    if ($action) {
        $where_conditions[] = "al.action = ?";
        $params[] = $action;
        $types .= "s";
    }
    
    if ($table_name) {
        $where_conditions[] = "al.table_name = ?";
        $params[] = $table_name;
        $types .= "s";
    }
    
    if ($date_from) {
        $where_conditions[] = "DATE(al.created_at) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }
    
    if ($date_to) {
        $where_conditions[] = "DATE(al.created_at) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    $sql = "SELECT al.*, u.role 
            FROM activity_logs al 
            LEFT JOIN users u ON al.user_id = u.id 
            $where_clause 
            ORDER BY al.created_at DESC 
            LIMIT ? OFFSET ?";
    
    $params[] = $limit;
    $params[] = $offset;
    $types .= "ii";
    
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    
    return $stmt->get_result();
}

// Fungsi untuk mendapatkan statistik aktivitas
function getActivityStats($conn, $user_id = null, $date_from = null, $date_to = null) {
    $where_conditions = [];
    $params = [];
    $types = "";
    
    if ($user_id) {
        $where_conditions[] = "user_id = ?";
        $params[] = $user_id;
        $types .= "i";
    }
    
    if ($date_from) {
        $where_conditions[] = "DATE(created_at) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }
    
    if ($date_to) {
        $where_conditions[] = "DATE(created_at) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    $sql = "SELECT 
                action,
                COUNT(*) as count,
                DATE(created_at) as date
            FROM activity_logs 
            $where_clause 
            GROUP BY action, DATE(created_at) 
            ORDER BY date DESC, count DESC";
    
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    
    return $stmt->get_result();
}

// Fungsi untuk mendapatkan jumlah total log
function getTotalLogs($conn, $user_id = null, $action = null, $table_name = null, $date_from = null, $date_to = null) {
    $where_conditions = [];
    $params = [];
    $types = "";
    
    if ($user_id) {
        $where_conditions[] = "user_id = ?";
        $params[] = $user_id;
        $types .= "i";
    }
    
    if ($action) {
        $where_conditions[] = "action = ?";
        $params[] = $action;
        $types .= "s";
    }
    
    if ($table_name) {
        $where_conditions[] = "table_name = ?";
        $params[] = $table_name;
        $types .= "s";
    }
    
    if ($date_from) {
        $where_conditions[] = "DATE(created_at) >= ?";
        $params[] = $date_from;
        $types .= "s";
    }
    
    if ($date_to) {
        $where_conditions[] = "DATE(created_at) <= ?";
        $params[] = $date_to;
        $types .= "s";
    }
    
    $where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";
    
    $sql = "SELECT COUNT(*) as total FROM activity_logs $where_clause";
    
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    
    return $row['total'];
}

// Fungsi untuk format waktu yang user-friendly
function formatActivityTime($timestamp) {
    $now = new DateTime();
    $time = new DateTime($timestamp);
    $diff = $now->diff($time);
    
    if ($diff->y > 0) {
        return $diff->y . ' tahun yang lalu';
    } elseif ($diff->m > 0) {
        return $diff->m . ' bulan yang lalu';
    } elseif ($diff->d > 0) {
        return $diff->d . ' hari yang lalu';
    } elseif ($diff->h > 0) {
        return $diff->h . ' jam yang lalu';
    } elseif ($diff->i > 0) {
        return $diff->i . ' menit yang lalu';
    } else {
        return 'Baru saja';
    }
}

// Fungsi untuk mendapatkan badge warna berdasarkan action
function getActionBadgeColor($action) {
    switch ($action) {
        case 'CREATE':
            return 'success';
        case 'UPDATE':
            return 'warning';
        case 'DELETE':
            return 'danger';
        case 'LOGIN':
            return 'info';
        case 'LOGOUT':
            return 'secondary';
        case 'UPLOAD':
            return 'primary';
        case 'DOWNLOAD':
            return 'info';
        case 'IMPORT':
            return 'success';
        case 'EXPORT':
            return 'info';
        default:
            return 'secondary';
    }
}

// Fungsi untuk mendapatkan icon berdasarkan action
function getActionIcon($action) {
    switch ($action) {
        case 'CREATE':
            return 'fas fa-plus';
        case 'UPDATE':
            return 'fas fa-edit';
        case 'DELETE':
            return 'fas fa-trash';
        case 'LOGIN':
            return 'fas fa-sign-in-alt';
        case 'LOGOUT':
            return 'fas fa-sign-out-alt';
        case 'UPLOAD':
            return 'fas fa-upload';
        case 'DOWNLOAD':
            return 'fas fa-download';
        case 'IMPORT':
            return 'fas fa-file-import';
        case 'EXPORT':
            return 'fas fa-file-export';
        default:
            return 'fas fa-info-circle';
    }
}
?> 