<?php
// Enable error reporting untuk debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Cek apakah file database ada
if (!file_exists('config/database.php')) {
    die("Error: File config/database.php tidak ditemukan. Pastikan file sudah diupload ke hosting.");
}

// Cek apakah file functions ada
if (!file_exists('includes/functions.php')) {
    die("Error: File includes/functions.php tidak ditemukan. Pastikan file sudah diupload ke hosting.");
}

// Include files dengan error handling
try {
    require_once 'config/database.php';
    require_once 'includes/functions.php';
} catch (Exception $e) {
    die("Error loading files: " . $e->getMessage());
}

// Redirect jika belum login
if (!isLoggedIn()) {
    header("Location: index.php");
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    try {
        $user_id = $_SESSION['user_id'];
        $sql = "SELECT full_name FROM users WHERE id = ?";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("i", $user_id);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $user = $result->fetch_assoc();
                $_SESSION['full_name'] = $user['full_name'];
            }
        }
    } catch (Exception $e) {
        error_log("Error fetching full_name: " . $e->getMessage());
    }
}

// Ambil data dokumen dengan error handling
$documents = [];
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';

try {
    if ($filter === 'all') {
        $sql = "SELECT d.*, u.username as uploader_name FROM documents d 
                JOIN users u ON d.uploaded_by = u.id 
                ORDER BY d.created_at DESC";
        $result = $conn->query($sql);
    } else {
        $sql = "SELECT d.*, u.username as uploader_name FROM documents d 
                JOIN users u ON d.uploaded_by = u.id 
                WHERE d.file_type = ? 
                ORDER BY d.created_at DESC";
        $stmt = $conn->prepare($sql);
        if ($stmt) {
            $stmt->bind_param("s", $filter);
            $stmt->execute();
            $result = $stmt->get_result();
        } else {
            $result = false;
        }
    }
    
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $documents[] = $row;
        }
    }
} catch (Exception $e) {
    error_log("Error fetching documents: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Aplikasi Persidangan Klasis Makassar</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/lightbox2@2.11.3/dist/css/lightbox.min.css" rel="stylesheet">
    <style>
        .navbar-brand img {
            border-radius: 5px;
        }
        .card {
            transition: transform 0.2s;
        }
        .card:hover {
            transform: translateY(-2px);
        }
        .file-icon {
            font-size: 2rem;
        }
        .filter-btn {
            border-radius: 20px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Aplikasi Persidangan Klasis Makassar
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link active" href="dashboard.php">
                            <i class="fas fa-home me-1"></i>Dashboard
                        </a>
                    </li>
                    <?php if (canUpload()): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="upload.php">
                            <i class="fas fa-upload me-1"></i>Upload Dokumen
                        </a>
                    </li>
                    <?php endif; ?>
                    <li class="nav-item">
                        <a class="nav-link" href="jemaat.php">
                            <i class="fas fa-church me-1"></i>Data Jemaat
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="peserta.php">
                            <i class="fas fa-users me-1"></i>Data Peserta
                        </a>
                    </li>
                    <?php if (hasRole('administrator') || hasRole('panitia')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="activity_logs.php">
                            <i class="fas fa-history me-1"></i>Log Aktivitas
                        </a>
                    </li>
                    <?php endif; ?>
                    <?php if (hasRole('administrator')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="users.php">
                            <i class="fas fa-users-cog me-1"></i>Kelola User
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="logout.php">
                                <i class="fas fa-sign-out-alt me-2"></i>Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-tachometer-alt me-2"></i>Dashboard</h2>
                    <div class="btn-group" role="group">
                        <a href="?filter=all" class="btn btn-outline-primary filter-btn <?php echo $filter === 'all' ? 'active' : ''; ?>">
                            <i class="fas fa-list me-1"></i>Semua
                        </a>
                        <a href="?filter=pdf" class="btn btn-outline-primary filter-btn <?php echo $filter === 'pdf' ? 'active' : ''; ?>">
                            <i class="fas fa-file-pdf me-1"></i>PDF
                        </a>
                        <a href="?filter=image" class="btn btn-outline-primary filter-btn <?php echo $filter === 'image' ? 'active' : ''; ?>">
                            <i class="fas fa-image me-1"></i>Gambar
                        </a>
                        <a href="?filter=video" class="btn btn-outline-primary filter-btn <?php echo $filter === 'video' ? 'active' : ''; ?>">
                            <i class="fas fa-video me-1"></i>Video
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <?php if (empty($documents)): ?>
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body text-center py-5">
                        <i class="fas fa-folder-open text-muted" style="font-size: 4rem;"></i>
                        <h4 class="mt-3 text-muted">Belum ada dokumen</h4>
                        <p class="text-muted">Mulai upload dokumen pertama Anda</p>
                        <?php if (canUpload()): ?>
                        <a href="upload.php" class="btn btn-primary">
                            <i class="fas fa-upload me-2"></i>Upload Dokumen
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php else: ?>
        <div class="row">
            <?php foreach ($documents as $doc): ?>
            <div class="col-md-6 col-lg-4 mb-4">
                <div class="card h-100">
                    <div class="card-body">
                        <div class="d-flex align-items-center mb-3">
                            <?php
                            $icon = 'fa-file';
                            $color = 'text-secondary';
                            if ($doc['file_type'] === 'pdf') {
                                $icon = 'fa-file-pdf';
                                $color = 'text-danger';
                            } elseif ($doc['file_type'] === 'image') {
                                $icon = 'fa-file-image';
                                $color = 'text-success';
                            } elseif ($doc['file_type'] === 'video') {
                                $icon = 'fa-file-video';
                                $color = 'text-warning';
                            }
                            ?>
                            <i class="fas <?php echo $icon; ?> file-icon <?php echo $color; ?> me-3"></i>
                            <div>
                                <h6 class="card-title mb-1"><?php echo htmlspecialchars($doc['title']); ?></h6>
                                <small class="text-muted"><?php echo htmlspecialchars($doc['uploader_name']); ?></small>
                            </div>
                        </div>
                        
                        <p class="card-text text-muted small">
                            <?php echo htmlspecialchars(substr($doc['description'], 0, 100)); ?>
                            <?php if (strlen($doc['description']) > 100): ?>...<?php endif; ?>
                        </p>
                        
                        <div class="d-flex justify-content-between align-items-center">
                            <small class="text-muted">
                                <i class="fas fa-calendar me-1"></i>
                                <?php echo date('d/m/Y H:i', strtotime($doc['created_at'])); ?>
                            </small>
                            <div class="btn-group btn-group-sm">
                                <?php if ($doc['file_type'] === 'image'): ?>
                                <a href="uploads/<?php echo $doc['filename']; ?>" class="btn btn-outline-primary" data-lightbox="gallery">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <?php elseif ($doc['file_type'] === 'pdf'): ?>
                                <a href="view_pdf.php?id=<?php echo $doc['id']; ?>" class="btn btn-outline-primary" target="_blank">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <?php elseif ($doc['file_type'] === 'video'): ?>
                                <a href="view_video.php?id=<?php echo $doc['id']; ?>" class="btn btn-outline-primary" target="_blank">
                                    <i class="fas fa-play"></i>
                                </a>
                                <?php endif; ?>
                                <a href="uploads/<?php echo $doc['filename']; ?>" class="btn btn-outline-success" download>
                                    <i class="fas fa-download"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/lightbox2@2.11.3/dist/js/lightbox.min.js"></script>
</body>
</html> 