<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login
if (!isLoggedIn()) {
    header("Location: index.php");
    exit();
}

// Ambil ID dokumen
$doc_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($doc_id <= 0) {
    header("Location: dashboard.php");
    exit();
}

// Ambil data dokumen
$sql = "SELECT d.*, u.username as uploader_name FROM documents d 
        JOIN users u ON d.uploaded_by = u.id 
        WHERE d.id = ? AND d.file_type = 'video'";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $doc_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header("Location: dashboard.php");
    exit();
}

$document = $result->fetch_assoc();
$video_path = 'uploads/' . $document['filename'];

if (!file_exists($video_path)) {
    header("Location: dashboard.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($document['title']); ?> - Video Player</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/plyr/3.7.8/plyr.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        .video-container {
            background: #000;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }
        .plyr {
            height: 70vh;
        }
        .video-info {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .video-controls {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
        }
        .video-controls .btn {
            margin: 0 5px;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .related-videos {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 20px;
        }
        .video-thumbnail {
            position: relative;
            overflow: hidden;
            border-radius: 8px;
            cursor: pointer;
            transition: transform 0.3s ease;
        }
        .video-thumbnail:hover {
            transform: scale(1.05);
        }
        .video-thumbnail .play-overlay {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0,0,0,0.7);
            color: white;
            border-radius: 50%;
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <i class="fas fa-file-alt me-2"></i>Sistem Dokumen Persidangan
            </a>
            
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="fas fa-arrow-left me-1"></i>Kembali ke Dashboard
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid mt-3">
        <!-- Video Info -->
        <div class="row mb-3">
            <div class="col-12">
                <div class="video-info">
                    <div class="row">
                        <div class="col-md-8">
                            <h4><i class="fas fa-play-circle me-2"></i><?php echo htmlspecialchars($document['title']); ?></h4>
                            <p class="mb-1"><?php echo htmlspecialchars($document['description']); ?></p>
                            <small>
                                <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($document['uploader_name']); ?> | 
                                <i class="fas fa-calendar me-1"></i><?php echo date('d/m/Y H:i', strtotime($document['created_at'])); ?> | 
                                <i class="fas fa-file me-1"></i><?php echo formatFileSize($document['file_size']); ?>
                            </small>
                        </div>
                        <div class="col-md-4 text-end">
                            <a href="uploads/<?php echo $document['filename']; ?>" class="btn btn-light" download>
                                <i class="fas fa-download me-2"></i>Download Video
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Video Player -->
        <div class="row">
            <div class="col-lg-9">
                <div class="card">
                    <div class="card-body p-0">
                        <div class="video-container">
                            <video id="player" playsinline controls>
                                <source src="<?php echo $video_path; ?>" type="video/mp4">
                                <source src="<?php echo $video_path; ?>" type="video/webm">
                                <source src="<?php echo $video_path; ?>" type="video/ogg">
                                Browser Anda tidak mendukung tag video.
                            </video>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Related Videos -->
            <div class="col-lg-3">
                <div class="related-videos">
                    <h5><i class="fas fa-video me-2"></i>Video Lainnya</h5>
                    <?php
                    // Ambil video lainnya
                    $sql = "SELECT d.*, u.username as uploader_name FROM documents d 
                            JOIN users u ON d.uploaded_by = u.id 
                            WHERE d.file_type = 'video' AND d.id != ? 
                            ORDER BY d.created_at DESC LIMIT 5";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("i", $doc_id);
                    $stmt->execute();
                    $related_result = $stmt->get_result();
                    
                    if ($related_result->num_rows > 0):
                        while ($related = $related_result->fetch_assoc()):
                    ?>
                    <div class="video-thumbnail mb-3" onclick="loadVideo(<?php echo $related['id']; ?>)">
                        <div class="play-overlay">
                            <i class="fas fa-play"></i>
                        </div>
                        <div class="p-2">
                            <h6 class="mb-1"><?php echo htmlspecialchars($related['title']); ?></h6>
                            <small class="text-muted">
                                <?php echo htmlspecialchars($related['uploader_name']); ?> • 
                                <?php echo date('d/m/Y', strtotime($related['created_at'])); ?>
                            </small>
                        </div>
                    </div>
                    <?php 
                        endwhile;
                    else:
                    ?>
                    <p class="text-muted">Tidak ada video lainnya</p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Video Controls -->
    <div class="video-controls">
        <button class="btn btn-primary" id="fullscreen-btn" title="Fullscreen">
            <i class="fas fa-expand"></i>
        </button>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/plyr/3.7.8/plyr.min.js"></script>
    <script>
        // Initialize Plyr
        const player = new Plyr('#player', {
            controls: [
                'play-large',
                'play',
                'progress',
                'current-time',
                'mute',
                'volume',
                'captions',
                'settings',
                'pip',
                'airplay',
                'fullscreen'
            ],
            settings: ['captions', 'quality', 'speed'],
            speed: { selected: 1, options: [0.5, 0.75, 1, 1.25, 1.5, 2] }
        });

        // Fullscreen button
        document.getElementById('fullscreen-btn').addEventListener('click', function() {
            if (player.fullscreen.active) {
                player.fullscreen.exit();
                this.innerHTML = '<i class="fas fa-expand"></i>';
            } else {
                player.fullscreen.enter();
                this.innerHTML = '<i class="fas fa-compress"></i>';
            }
        });

        // Load related video
        function loadVideo(videoId) {
            window.location.href = 'view_video.php?id=' + videoId;
        }

        // Keyboard shortcuts
        document.addEventListener('keydown', function(e) {
            switch(e.key) {
                case ' ':
                    e.preventDefault();
                    player.togglePlay();
                    break;
                case 'ArrowLeft':
                    player.rewind(10);
                    break;
                case 'ArrowRight':
                    player.forward(10);
                    break;
                case 'ArrowUp':
                    player.increaseVolume(0.1);
                    break;
                case 'ArrowDown':
                    player.decreaseVolume(0.1);
                    break;
                case 'f':
                    if (player.fullscreen.active) {
                        player.fullscreen.exit();
                    } else {
                        player.fullscreen.enter();
                    }
                    break;
            }
        });

        // Auto-hide controls on mobile
        let hideTimeout;
        function showControls() {
            clearTimeout(hideTimeout);
            document.querySelector('.plyr__controls').style.opacity = '1';
            hideTimeout = setTimeout(() => {
                if (!player.playing) return;
                document.querySelector('.plyr__controls').style.opacity = '0';
            }, 3000);
        }

        document.addEventListener('mousemove', showControls);
        document.addEventListener('touchstart', showControls);
    </script>
</body>
</html> 