<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login
if (!isLoggedIn()) {
    header("Location: index.php");
    exit();
}

// Ambil ID dokumen
$doc_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($doc_id <= 0) {
    header("Location: dashboard.php");
    exit();
}

// Ambil data dokumen
$sql = "SELECT d.*, u.username as uploader_name FROM documents d 
        JOIN users u ON d.uploaded_by = u.id 
        WHERE d.id = ? AND d.file_type = 'pdf'";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $doc_id);
$stmt->execute();
$result = $stmt->get_result();

if ($result->num_rows === 0) {
    header("Location: dashboard.php");
    exit();
}

$document = $result->fetch_assoc();
$pdf_path = 'uploads/' . $document['filename'];

if (!file_exists($pdf_path)) {
    header("Location: dashboard.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($document['title']); ?> - PDF Viewer</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf_viewer.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
    <style>
        #pdf-container {
            height: 80vh;
            border: 1px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
        }
        .pdf-toolbar {
            background: #f8f9fa;
            padding: 10px;
            border-bottom: 1px solid #ddd;
        }
        .pdf-viewer {
            height: calc(100% - 60px);
            overflow: auto;
        }
        .page-container {
            margin: 10px auto;
            text-align: center;
        }
        .page-canvas {
            border: 1px solid #ccc;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .loading {
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100%;
            font-size: 18px;
            color: #666;
        }
        .flip-controls {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1000;
        }
        .flip-controls .btn {
            margin: 0 5px;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <i class="fas fa-file-alt me-2"></i>Sistem Dokumen Persidangan
            </a>
            
            <div class="navbar-nav ms-auto">
                <a class="nav-link" href="dashboard.php">
                    <i class="fas fa-arrow-left me-1"></i>Kembali ke Dashboard
                </a>
            </div>
        </div>
    </nav>

    <div class="container-fluid mt-3">
        <!-- Document Info -->
        <div class="row mb-3">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-8">
                                <h4><i class="fas fa-file-pdf text-danger me-2"></i><?php echo htmlspecialchars($document['title']); ?></h4>
                                <p class="text-muted mb-1"><?php echo htmlspecialchars($document['description']); ?></p>
                                <small class="text-muted">
                                    <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($document['uploader_name']); ?> | 
                                    <i class="fas fa-calendar me-1"></i><?php echo date('d/m/Y H:i', strtotime($document['created_at'])); ?> | 
                                    <i class="fas fa-file me-1"></i><?php echo formatFileSize($document['file_size']); ?>
                                </small>
                            </div>
                            <div class="col-md-4 text-end">
                                <a href="uploads/<?php echo $document['filename']; ?>" class="btn btn-outline-primary" download>
                                    <i class="fas fa-download me-2"></i>Download PDF
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- PDF Viewer -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body p-0">
                        <div id="pdf-container">
                            <div class="pdf-toolbar">
                                <div class="row align-items-center">
                                    <div class="col-md-6">
                                        <div class="btn-group" role="group">
                                            <button type="button" class="btn btn-outline-secondary btn-sm" id="prev-page">
                                                <i class="fas fa-chevron-left"></i> Sebelumnya
                                            </button>
                                            <button type="button" class="btn btn-outline-secondary btn-sm" id="next-page">
                                                Berikutnya <i class="fas fa-chevron-right"></i>
                                            </button>
                                        </div>
                                        <span class="ms-3">
                                            Halaman <span id="page-num"></span> dari <span id="page-count"></span>
                                        </span>
                                    </div>
                                    <div class="col-md-6 text-end">
                                        <div class="btn-group" role="group">
                                            <button type="button" class="btn btn-outline-secondary btn-sm" id="zoom-out">
                                                <i class="fas fa-search-minus"></i>
                                            </button>
                                            <span class="btn btn-outline-secondary btn-sm" id="zoom-level">100%</span>
                                            <button type="button" class="btn btn-outline-secondary btn-sm" id="zoom-in">
                                                <i class="fas fa-search-plus"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="pdf-viewer" id="pdf-viewer">
                                <div class="loading">
                                    <div class="text-center">
                                        <div class="spinner-border text-primary mb-3" role="status">
                                            <span class="visually-hidden">Loading...</span>
                                        </div>
                                        <p>Memuat PDF...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Flip Controls -->
    <div class="flip-controls">
        <button class="btn btn-primary" id="fullscreen-btn" title="Fullscreen">
            <i class="fas fa-expand"></i>
        </button>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.min.js"></script>
    <script>
        // Set worker path
        pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/3.11.174/pdf.worker.min.js';

        let pdfDoc = null;
        let pageNum = 1;
        let pageRendering = false;
        let pageNumPending = null;
        let scale = 1.0;
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');

        // Load PDF
        pdfjsLib.getDocument('<?php echo $pdf_path; ?>').promise.then(function(pdf) {
            pdfDoc = pdf;
            document.getElementById('page-count').textContent = pdf.numPages;
            
            // Initial render
            renderPage(pageNum);
        }).catch(function(error) {
            console.error('Error loading PDF:', error);
            document.getElementById('pdf-viewer').innerHTML = `
                <div class="loading">
                    <div class="text-center">
                        <i class="fas fa-exclamation-triangle fa-3x text-warning mb-3"></i>
                        <p>Gagal memuat PDF</p>
                    </div>
                </div>
            `;
        });

        function renderPage(num) {
            pageRendering = true;
            
            pdfDoc.getPage(num).then(function(page) {
                const viewport = page.getViewport({scale: scale});
                canvas.height = viewport.height;
                canvas.width = viewport.width;

                const renderContext = {
                    canvasContext: ctx,
                    viewport: viewport
                };
                
                const renderTask = page.render(renderContext);

                renderTask.promise.then(function() {
                    pageRendering = false;
                    if (pageNumPending !== null) {
                        renderPage(pageNumPending);
                        pageNumPending = null;
                    }
                });
            });

            document.getElementById('page-num').textContent = num;
        }

        function queueRenderPage(num) {
            if (pageRendering) {
                pageNumPending = num;
            } else {
                renderPage(num);
            }
        }

        function onPrevPage() {
            if (pageNum <= 1) {
                return;
            }
            pageNum--;
            queueRenderPage(pageNum);
        }

        function onNextPage() {
            if (pageNum >= pdfDoc.numPages) {
                return;
            }
            pageNum++;
            queueRenderPage(pageNum);
        }

        function onZoomIn() {
            scale += 0.2;
            document.getElementById('zoom-level').textContent = Math.round(scale * 100) + '%';
            queueRenderPage(pageNum);
        }

        function onZoomOut() {
            if (scale <= 0.5) return;
            scale -= 0.2;
            document.getElementById('zoom-level').textContent = Math.round(scale * 100) + '%';
            queueRenderPage(pageNum);
        }

        // Event listeners
        document.getElementById('prev-page').addEventListener('click', onPrevPage);
        document.getElementById('next-page').addEventListener('click', onNextPage);
        document.getElementById('zoom-in').addEventListener('click', onZoomIn);
        document.getElementById('zoom-out').addEventListener('click', onZoomOut);

        // Keyboard navigation
        document.addEventListener('keydown', function(e) {
            if (e.key === 'ArrowLeft') {
                onPrevPage();
            } else if (e.key === 'ArrowRight') {
                onNextPage();
            } else if (e.key === '+') {
                onZoomIn();
            } else if (e.key === '-') {
                onZoomOut();
            }
        });

        // Fullscreen
        document.getElementById('fullscreen-btn').addEventListener('click', function() {
            const container = document.getElementById('pdf-container');
            if (!document.fullscreenElement) {
                container.requestFullscreen();
                this.innerHTML = '<i class="fas fa-compress"></i>';
            } else {
                document.exitFullscreen();
                this.innerHTML = '<i class="fas fa-expand"></i>';
            }
        });

        // Append canvas to viewer
        document.getElementById('pdf-viewer').appendChild(canvas);
    </script>
</body>
</html> 