<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login atau tidak punya akses upload
if (!isLoggedIn() || !canUpload()) {
    header("Location: dashboard.php");
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    $user_id = $_SESSION['user_id'];
    $sql = "SELECT full_name FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $_SESSION['full_name'] = $user['full_name'];
    }
}

$success = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $title = trim($_POST['title']);
    $description = trim($_POST['description']);
    $upload_type = $_POST['upload_type'] ?? 'single';
    
    if (empty($title)) {
        $error = 'Judul dokumen harus diisi!';
    } else {
        if ($upload_type === 'single') {
            // Single file upload
            if (!isset($_FILES['document']) || $_FILES['document']['error'] !== UPLOAD_ERR_OK) {
                $error = 'Pilih file dokumen yang valid!';
            } else {
                $file = $_FILES['document'];
                $original_filename = $file['name'];
                $file_size = $file['size'];
                $tmp_name = $file['tmp_name'];
                
                // Validasi tipe file
                if (!isAllowedFileType($original_filename)) {
                    $error = 'Tipe file tidak diizinkan! File yang diizinkan: PDF, JPG, JPEG, PNG, GIF, MP4, AVI, MOV, WMV';
                } elseif ($file_size > 100 * 1024 * 1024) { // 100MB limit
                    $error = 'Ukuran file terlalu besar! Maksimal 100MB';
                } else {
                    // Buat direktori uploads jika belum ada
                    $upload_dir = 'uploads/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0777, true);
                    }
                    
                    // Generate nama file unik
                    $filename = generateUniqueFilename($original_filename);
                    $file_path = $upload_dir . $filename;
                    
                    // Upload file
                    if (move_uploaded_file($tmp_name, $file_path)) {
                        // Tentukan tipe file
                        $file_type = getFileCategory($original_filename);
                        
                        // Simpan ke database
                        $sql = "INSERT INTO documents (title, description, filename, original_filename, file_type, file_size, uploaded_by) VALUES (?, ?, ?, ?, ?, ?, ?)";
                        $stmt = $conn->prepare($sql);
                        $stmt->bind_param("sssssii", $title, $description, $filename, $original_filename, $file_type, $file_size, $_SESSION['user_id']);
                        
                        if ($stmt->execute()) {
                            $document_id = $conn->insert_id;
                            $success = 'Dokumen berhasil diupload!';
                            
                            // Log aktivitas UPLOAD
                            logActivity($conn, 'UPLOAD', 'documents', $document_id, "Upload dokumen: $title ($original_filename)");
                            
                            // Reset form
                            $title = $description = '';
                        } else {
                            $error = 'Gagal menyimpan data ke database!';
                            // Hapus file yang sudah diupload
                            unlink($file_path);
                        }
                    } else {
                        $error = 'Gagal mengupload file!';
                    }
                }
            }
        } elseif ($upload_type === 'multiple_images') {
            // Multiple images upload
            if (!isset($_FILES['images']) || empty($_FILES['images']['name'][0])) {
                $error = 'Pilih minimal satu file gambar!';
            } else {
                $files = $_FILES['images'];
                $file_count = count($files['name']);
                
                if ($file_count > 10) {
                    $error = 'Maksimal 10 file gambar sekaligus!';
                } else {
                    // Buat direktori uploads jika belum ada
                    $upload_dir = 'uploads/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0777, true);
                    }
                    
                    $uploaded_count = 0;
                    $errors = [];
                    
                    for ($i = 0; $i < $file_count; $i++) {
                        if ($files['error'][$i] === UPLOAD_ERR_OK) {
                            $original_filename = $files['name'][$i];
                            $file_size = $files['size'][$i];
                            $tmp_name = $files['tmp_name'][$i];
                            
                            // Validasi tipe file
                            if (!isImage($original_filename)) {
                                $errors[] = "File '$original_filename' bukan file gambar yang valid!";
                                continue;
                            } elseif ($file_size > 100 * 1024 * 1024) { // 100MB limit
                                $errors[] = "File '$original_filename' terlalu besar! Maksimal 100MB";
                                continue;
                            }
                            
                            // Generate nama file unik
                            $filename = generateUniqueFilename($original_filename);
                            $file_path = $upload_dir . $filename;
                            
                            // Upload file
                            if (move_uploaded_file($tmp_name, $file_path)) {
                                // Simpan ke database
                                $sql = "INSERT INTO documents (title, description, filename, original_filename, file_type, file_size, uploaded_by) VALUES (?, ?, ?, ?, 'image', ?, ?)";
                                $stmt = $conn->prepare($sql);
                                $image_title = $title . ' - ' . pathinfo($original_filename, PATHINFO_FILENAME);
                                $stmt->bind_param("ssssii", $image_title, $description, $filename, $original_filename, $file_size, $_SESSION['user_id']);
                                
                                if ($stmt->execute()) {
                                    $uploaded_count++;
                                } else {
                                    $errors[] = "Gagal menyimpan '$original_filename' ke database!";
                                    unlink($file_path);
                                }
                            } else {
                                $errors[] = "Gagal mengupload '$original_filename'!";
                            }
                        }
                    }
                    
                    if ($uploaded_count > 0) {
                        $success = "Berhasil mengupload $uploaded_count file gambar!";
                        if (!empty($errors)) {
                            $success .= " Beberapa file gagal: " . implode(', ', $errors);
                        }
                        // Reset form
                        $title = $description = '';
                    } else {
                        $error = 'Gagal mengupload semua file! ' . implode(', ', $errors);
                    }
                }
            }
        } elseif ($upload_type === 'multiple_videos') {
            // Multiple videos upload
            if (!isset($_FILES['videos']) || empty($_FILES['videos']['name'][0])) {
                $error = 'Pilih minimal satu file video!';
            } else {
                $files = $_FILES['videos'];
                $file_count = count($files['name']);
                
                if ($file_count > 5) {
                    $error = 'Maksimal 5 file video sekaligus!';
                } else {
                    // Buat direktori uploads jika belum ada
                    $upload_dir = 'uploads/';
                    if (!is_dir($upload_dir)) {
                        mkdir($upload_dir, 0777, true);
                    }
                    
                    $uploaded_count = 0;
                    $errors = [];
                    
                    for ($i = 0; $i < $file_count; $i++) {
                        if ($files['error'][$i] === UPLOAD_ERR_OK) {
                            $original_filename = $files['name'][$i];
                            $file_size = $files['size'][$i];
                            $tmp_name = $files['tmp_name'][$i];
                            
                            // Validasi tipe file
                            if (!isVideo($original_filename)) {
                                $errors[] = "File '$original_filename' bukan file video yang valid!";
                                continue;
                            } elseif ($file_size > 100 * 1024 * 1024) { // 100MB limit
                                $errors[] = "File '$original_filename' terlalu besar! Maksimal 100MB";
                                continue;
                            }
                            
                            // Generate nama file unik
                            $filename = generateUniqueFilename($original_filename);
                            $file_path = $upload_dir . $filename;
                            
                            // Upload file
                            if (move_uploaded_file($tmp_name, $file_path)) {
                                // Simpan ke database
                                $sql = "INSERT INTO documents (title, description, filename, original_filename, file_type, file_size, uploaded_by) VALUES (?, ?, ?, ?, 'video', ?, ?)";
                                $stmt = $conn->prepare($sql);
                                $video_title = $title . ' - ' . pathinfo($original_filename, PATHINFO_FILENAME);
                                $stmt->bind_param("ssssii", $video_title, $description, $filename, $original_filename, $file_size, $_SESSION['user_id']);
                                
                                if ($stmt->execute()) {
                                    $uploaded_count++;
                                } else {
                                    $errors[] = "Gagal menyimpan '$original_filename' ke database!";
                                    unlink($file_path);
                                }
                            } else {
                                $errors[] = "Gagal mengupload '$original_filename'!";
                            }
                        }
                    }
                    
                    if ($uploaded_count > 0) {
                        $success = "Berhasil mengupload $uploaded_count file video!";
                        if (!empty($errors)) {
                            $success .= " Beberapa file gagal: " . implode(', ', $errors);
                        }
                        // Reset form
                        $title = $description = '';
                    } else {
                        $error = 'Gagal mengupload semua file! ' . implode(', ', $errors);
                    }
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Upload Dokumen - Sistem Manajemen Dokumen</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Sistem Dokumen Persidangan
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-home me-1"></i>Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="upload.php">
                            <i class="fas fa-upload me-1"></i>Upload Dokumen
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jemaat.php">
                            <i class="fas fa-church me-1"></i>Data Jemaat
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="peserta.php">
                            <i class="fas fa-users me-1"></i>Data Peserta
                        </a>
                    </li>
                    <?php if (hasRole('administrator') || hasRole('panitia')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="activity_logs.php">
                            <i class="fas fa-history me-1"></i>Log Aktivitas
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="profile.php">
                                <i class="fas fa-user-edit me-2"></i>Profile
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php">
                                <i class="fas fa-sign-out-alt me-2"></i>Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card shadow">
                    <div class="card-header bg-primary text-white">
                        <h4 class="mb-0">
                            <i class="fas fa-upload me-2"></i>Upload Dokumen Baru
                        </h4>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($error): ?>
                            <div class="alert alert-danger">
                                <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" enctype="multipart/form-data" id="uploadForm">
                            <div class="mb-3">
                                <label for="title" class="form-label">Judul Dokumen <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="title" name="title" value="<?php echo htmlspecialchars($title ?? ''); ?>" required>
                            </div>
                            
                            <div class="mb-3">
                                <label for="description" class="form-label">Deskripsi</label>
                                <textarea class="form-control" id="description" name="description" rows="3"><?php echo htmlspecialchars($description ?? ''); ?></textarea>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Tipe Upload</label>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="upload_type" id="single_upload" value="single" checked>
                                    <label class="form-check-label" for="single_upload">
                                        <i class="fas fa-file me-1"></i>Upload Tunggal (PDF/Gambar/Video)
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="upload_type" id="multiple_images" value="multiple_images">
                                    <label class="form-check-label" for="multiple_images">
                                        <i class="fas fa-images me-1"></i>Upload Multiple Gambar
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="upload_type" id="multiple_videos" value="multiple_videos">
                                    <label class="form-check-label" for="multiple_videos">
                                        <i class="fas fa-video me-1"></i>Upload Multiple Video
                                    </label>
                                </div>
                            </div>
                            
                            <!-- Single Upload -->
                            <div id="single_upload_section" class="mb-4">
                                <label for="document" class="form-label">Pilih File <span class="text-danger">*</span></label>
                                <input type="file" class="form-control" id="document" name="document" accept=".pdf,.jpg,.jpeg,.png,.gif,.mp4,.avi,.mov,.wmv">
                                <div class="form-text">
                                    <strong>Tipe file yang diizinkan:</strong><br>
                                    • <i class="fas fa-file-pdf text-danger"></i> PDF (untuk flipping book)<br>
                                    • <i class="fas fa-file-image text-success"></i> Gambar: JPG, JPEG, PNG, GIF (untuk gallery)<br>
                                    • <i class="fas fa-file-video text-primary"></i> Video: MP4, AVI, MOV, WMV (untuk showcase)<br>
                                    <strong>Ukuran maksimal: 100MB per file</strong>
                                </div>
                            </div>
                            
                            <!-- Multiple Images Upload -->
                            <div id="multiple_images_section" class="mb-4" style="display: none;">
                                <label for="images" class="form-label">Pilih Multiple Gambar <span class="text-danger">*</span></label>
                                <input type="file" class="form-control" id="images" name="images[]" accept=".jpg,.jpeg,.png,.gif" multiple>
                                <div class="form-text">
                                    <strong>Tipe file yang diizinkan:</strong> JPG, JPEG, PNG, GIF<br>
                                    <strong>Ukuran maksimal: 100MB per file</strong><br>
                                    <strong>Maksimal: 10 file sekaligus</strong>
                                </div>
                                <div id="image_preview_container" class="mt-3"></div>
                            </div>
                            
                            <!-- Multiple Videos Upload -->
                            <div id="multiple_videos_section" class="mb-4" style="display: none;">
                                <label for="videos" class="form-label">Pilih Multiple Video <span class="text-danger">*</span></label>
                                <input type="file" class="form-control" id="videos" name="videos[]" accept=".mp4,.avi,.mov,.wmv" multiple>
                                <div class="form-text">
                                    <strong>Tipe file yang diizinkan:</strong> MP4, AVI, MOV, WMV<br>
                                    <strong>Ukuran maksimal: 100MB per file</strong><br>
                                    <strong>Maksimal: 5 file sekaligus</strong>
                                </div>
                                <div id="video_preview_container" class="mt-3"></div>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-upload me-2"></i>Upload Dokumen
                                </button>
                                <a href="dashboard.php" class="btn btn-outline-secondary">
                                    <i class="fas fa-arrow-left me-2"></i>Kembali ke Dashboard
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Preview Area -->
                <div class="card mt-4">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-eye me-2"></i>Preview File
                        </h5>
                    </div>
                    <div class="card-body">
                        <div id="filePreview" class="text-center">
                            <p class="text-muted">Pilih file untuk melihat preview</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle upload sections based on radio button selection
        document.querySelectorAll('input[name="upload_type"]').forEach(function(radio) {
            radio.addEventListener('change', function() {
                // Hide all sections
                document.getElementById('single_upload_section').style.display = 'none';
                document.getElementById('multiple_images_section').style.display = 'none';
                document.getElementById('multiple_videos_section').style.display = 'none';
                
                // Show selected section
                if (this.value === 'single') {
                    document.getElementById('single_upload_section').style.display = 'block';
                } else if (this.value === 'multiple_images') {
                    document.getElementById('multiple_images_section').style.display = 'block';
                } else if (this.value === 'multiple_videos') {
                    document.getElementById('multiple_videos_section').style.display = 'block';
                }
                
                // Clear preview
                document.getElementById('filePreview').innerHTML = '<p class="text-muted">Pilih file untuk melihat preview</p>';
                document.getElementById('image_preview_container').innerHTML = '';
                document.getElementById('video_preview_container').innerHTML = '';
            });
        });

        // Single file preview
        document.getElementById('document').addEventListener('change', function(e) {
            const file = e.target.files[0];
            const preview = document.getElementById('filePreview');
            
            if (file) {
                const reader = new FileReader();
                const fileType = file.type;
                
                if (fileType.startsWith('image/')) {
                    reader.onload = function(e) {
                        preview.innerHTML = `
                            <img src="${e.target.result}" class="img-fluid" style="max-height: 300px;" alt="Preview">
                            <p class="mt-2"><strong>${file.name}</strong> (${(file.size / 1024 / 1024).toFixed(2)} MB)</p>
                        `;
                    };
                    reader.readAsDataURL(file);
                } else if (fileType === 'application/pdf') {
                    preview.innerHTML = `
                        <i class="fas fa-file-pdf fa-5x text-danger mb-3"></i>
                        <p><strong>${file.name}</strong> (${(file.size / 1024 / 1024).toFixed(2)} MB)</p>
                        <p class="text-muted">PDF akan ditampilkan sebagai flipping book</p>
                    `;
                } else if (fileType.startsWith('video/')) {
                    preview.innerHTML = `
                        <i class="fas fa-file-video fa-5x text-primary mb-3"></i>
                        <p><strong>${file.name}</strong> (${(file.size / 1024 / 1024).toFixed(2)} MB)</p>
                        <p class="text-muted">Video akan ditampilkan dalam showcase</p>
                    `;
                } else {
                    preview.innerHTML = `
                        <i class="fas fa-file fa-5x text-secondary mb-3"></i>
                        <p><strong>${file.name}</strong> (${(file.size / 1024 / 1024).toFixed(2)} MB)</p>
                    `;
                }
            } else {
                preview.innerHTML = '<p class="text-muted">Pilih file untuk melihat preview</p>';
            }
        });

        // Multiple images preview
        document.getElementById('images').addEventListener('change', function(e) {
            const files = e.target.files;
            const container = document.getElementById('image_preview_container');
            const preview = document.getElementById('filePreview');
            
            if (files.length > 0) {
                let previewHTML = '<div class="row">';
                let containerHTML = '<div class="row">';
                
                for (let i = 0; i < files.length; i++) {
                    const file = files[i];
                    const reader = new FileReader();
                    
                    reader.onload = function(e) {
                        const fileIndex = i;
                        containerHTML += `
                            <div class="col-md-3 mb-3">
                                <div class="card">
                                    <img src="${e.target.result}" class="card-img-top" style="height: 150px; object-fit: cover;" alt="Preview">
                                    <div class="card-body p-2">
                                        <small class="text-muted">${file.name}</small><br>
                                        <small class="text-muted">${(file.size / 1024 / 1024).toFixed(2)} MB</small>
                                    </div>
                                </div>
                            </div>
                        `;
                        
                        if (fileIndex === files.length - 1) {
                            container.innerHTML = containerHTML + '</div>';
                        }
                    };
                    
                    reader.readAsDataURL(file);
                }
                
                preview.innerHTML = `
                    <i class="fas fa-images fa-3x text-success mb-3"></i>
                    <p><strong>${files.length} file gambar dipilih</strong></p>
                    <p class="text-muted">Total ukuran: ${(Array.from(files).reduce((acc, file) => acc + file.size, 0) / 1024 / 1024).toFixed(2)} MB</p>
                `;
            } else {
                container.innerHTML = '';
                preview.innerHTML = '<p class="text-muted">Pilih file untuk melihat preview</p>';
            }
        });

        // Multiple videos preview
        document.getElementById('videos').addEventListener('change', function(e) {
            const files = e.target.files;
            const container = document.getElementById('video_preview_container');
            const preview = document.getElementById('filePreview');
            
            if (files.length > 0) {
                let containerHTML = '<div class="row">';
                
                for (let i = 0; i < files.length; i++) {
                    const file = files[i];
                    containerHTML += `
                        <div class="col-md-6 mb-3">
                            <div class="card">
                                <div class="card-body text-center">
                                    <i class="fas fa-file-video fa-3x text-primary mb-2"></i>
                                    <h6 class="card-title">${file.name}</h6>
                                    <p class="card-text text-muted">${(file.size / 1024 / 1024).toFixed(2)} MB</p>
                                </div>
                            </div>
                        </div>
                    `;
                }
                
                container.innerHTML = containerHTML + '</div>';
                
                preview.innerHTML = `
                    <i class="fas fa-video fa-3x text-primary mb-3"></i>
                    <p><strong>${files.length} file video dipilih</strong></p>
                    <p class="text-muted">Total ukuran: ${(Array.from(files).reduce((acc, file) => acc + file.size, 0) / 1024 / 1024).toFixed(2)} MB</p>
                `;
            } else {
                container.innerHTML = '';
                preview.innerHTML = '<p class="text-muted">Pilih file untuk melihat preview</p>';
            }
        });

        // Form validation
        document.getElementById('uploadForm').addEventListener('submit', function(e) {
            const uploadType = document.querySelector('input[name="upload_type"]:checked').value;
            let hasFile = false;
            
            if (uploadType === 'single') {
                hasFile = document.getElementById('document').files.length > 0;
            } else if (uploadType === 'multiple_images') {
                hasFile = document.getElementById('images').files.length > 0;
            } else if (uploadType === 'multiple_videos') {
                hasFile = document.getElementById('videos').files.length > 0;
            }
            
            if (!hasFile) {
                e.preventDefault();
                alert('Pilih file terlebih dahulu!');
            }
        });
    </script>
</body>
</html> 