<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login atau bukan admin/panitia
if (!isLoggedIn() || !canUpload()) {
    header("Location: dashboard.php");
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    $user_id = $_SESSION['user_id'];
    $sql = "SELECT full_name FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $_SESSION['full_name'] = $user['full_name'];
    }
}

$success = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        if ($_POST['action'] == 'add') {
            $nama_jemaat = trim($_POST['nama_jemaat']);
            $alamat_jemaat = trim($_POST['alamat_jemaat']);
            $kota = trim($_POST['kota']);
            $provinsi = trim($_POST['provinsi']);
            $kode_pos = trim($_POST['kode_pos']);
            $nomor_telepon = trim($_POST['nomor_telepon']);
            $email = trim($_POST['email']);
            $nama_pendeta = trim($_POST['nama_pendeta']);
            $jumlah_anggota = (int)$_POST['jumlah_anggota'];
            $deskripsi = trim($_POST['deskripsi']);
            
            if (empty($nama_jemaat)) {
                $error = 'Nama jemaat harus diisi!';
            } else {
                $sql = "INSERT INTO jemaat (nama_jemaat, alamat_jemaat, kota, provinsi, kode_pos, nomor_telepon, email, nama_pendeta, jumlah_anggota, deskripsi) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssssssis", $nama_jemaat, $alamat_jemaat, $kota, $provinsi, $kode_pos, $nomor_telepon, $email, $nama_pendeta, $jumlah_anggota, $deskripsi);
                
                if ($stmt->execute()) {
                    $jemaat_id = $conn->insert_id;
                    $success = 'Data jemaat berhasil ditambahkan!';
                    
                    // Log aktivitas CREATE
                    logActivity($conn, 'CREATE', 'jemaat', $jemaat_id, "Menambahkan jemaat: $nama_jemaat");
                    
                    // Reset form
                    $nama_jemaat = $alamat_jemaat = $kota = $provinsi = $kode_pos = $nomor_telepon = $email = $nama_pendeta = $deskripsi = '';
                    $jumlah_anggota = 0;
                } else {
                    $error = 'Gagal menambahkan data jemaat!';
                }
            }
        } elseif ($_POST['action'] == 'edit') {
            $jemaat_id = (int)$_POST['jemaat_id'];
            $nama_jemaat = trim($_POST['nama_jemaat']);
            $alamat_jemaat = trim($_POST['alamat_jemaat']);
            $kota = trim($_POST['kota']);
            $provinsi = trim($_POST['provinsi']);
            $kode_pos = trim($_POST['kode_pos']);
            $nomor_telepon = trim($_POST['nomor_telepon']);
            $email = trim($_POST['email']);
            $nama_pendeta = trim($_POST['nama_pendeta']);
            $jumlah_anggota = (int)$_POST['jumlah_anggota'];
            $deskripsi = trim($_POST['deskripsi']);
            
            if (empty($nama_jemaat)) {
                $error = 'Nama jemaat harus diisi!';
            } else {
                $sql = "UPDATE jemaat SET nama_jemaat = ?, alamat_jemaat = ?, kota = ?, provinsi = ?, kode_pos = ?, nomor_telepon = ?, email = ?, nama_pendeta = ?, jumlah_anggota = ?, deskripsi = ? WHERE id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssssssssisi", $nama_jemaat, $alamat_jemaat, $kota, $provinsi, $kode_pos, $nomor_telepon, $email, $nama_pendeta, $jumlah_anggota, $deskripsi, $jemaat_id);
                
                if ($stmt->execute()) {
                    $success = 'Data jemaat berhasil diupdate!';
                    
                    // Log aktivitas UPDATE
                    logActivity($conn, 'UPDATE', 'jemaat', $jemaat_id, "Mengupdate jemaat: $nama_jemaat");
                } else {
                    $error = 'Gagal mengupdate data jemaat!';
                }
            }
        } elseif ($_POST['action'] == 'delete') {
            $jemaat_id = (int)$_POST['jemaat_id'];
            
            // Cek apakah ada peserta yang terkait
            $check_sql = "SELECT COUNT(*) as count FROM peserta WHERE jemaat_id = ?";
            $check_stmt = $conn->prepare($check_sql);
            $check_stmt->bind_param("i", $jemaat_id);
            $check_stmt->execute();
            $result = $check_stmt->get_result();
            $count = $result->fetch_assoc()['count'];
            
            if ($count > 0) {
                $error = "Tidak dapat menghapus jemaat karena masih ada $count peserta yang terkait!";
            } else {
                $sql = "DELETE FROM jemaat WHERE id = ?";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("i", $jemaat_id);
                
                if ($stmt->execute()) {
                    $success = 'Data jemaat berhasil dihapus!';
                    
                    // Log aktivitas DELETE
                    logActivity($conn, 'DELETE', 'jemaat', $jemaat_id, "Menghapus jemaat dengan ID: $jemaat_id");
                } else {
                    $error = 'Gagal menghapus data jemaat!';
                }
            }
        }
    }
}

// Ambil data jemaat
$sql = "SELECT j.*, COUNT(p.id) as jumlah_peserta FROM jemaat j LEFT JOIN peserta p ON j.id = p.jemaat_id GROUP BY j.id ORDER BY j.created_at DESC";
$result = $conn->query($sql);
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Kelola Data Jemaat - Sistem Manajemen Dokumen</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Sistem Dokumen Persidangan
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="dashboard.php">
                            <i class="fas fa-home me-1"></i>Dashboard
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="upload.php">
                            <i class="fas fa-upload me-1"></i>Upload Dokumen
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="jemaat.php">
                            <i class="fas fa-church me-1"></i>Data Jemaat
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="peserta.php">
                            <i class="fas fa-users me-1"></i>Data Peserta
                        </a>
                    </li>
                    <?php if (hasRole('administrator') || hasRole('panitia')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="activity_logs.php">
                            <i class="fas fa-history me-1"></i>Log Aktivitas
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="profile.php">
                                <i class="fas fa-user-edit me-2"></i>Profile
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php">
                                <i class="fas fa-sign-out-alt me-2"></i>Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Header -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h2><i class="fas fa-church me-2"></i>Kelola Data Jemaat</h2>
                <p class="text-muted">Kelola data jemaat dan informasi gereja</p>
            </div>
            <div class="col-md-4 text-end">
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addJemaatModal">
                    <i class="fas fa-plus me-2"></i>Tambah Jemaat
                </button>
            </div>
        </div>

        <!-- Alerts -->
        <?php if ($success): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i><?php echo $success; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="fas fa-exclamation-circle me-2"></i><?php echo $error; ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Jemaat Table -->
        <div class="card">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>ID</th>
                                <th>Nama Jemaat</th>
                                <th>Alamat</th>
                                <th>Kota</th>
                                <th>Pendeta</th>
                                <th>Jumlah Anggota</th>
                                <th>Jumlah Peserta</th>
                                <th>Telepon</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if ($result->num_rows > 0): ?>
                                <?php while ($jemaat = $result->fetch_assoc()): ?>
                                <tr>
                                    <td><?php echo $jemaat['id']; ?></td>
                                    <td>
                                        <strong><?php echo htmlspecialchars($jemaat['nama_jemaat']); ?></strong>
                                    </td>
                                    <td><?php echo htmlspecialchars($jemaat['alamat_jemaat']); ?></td>
                                    <td><?php echo htmlspecialchars($jemaat['kota']); ?></td>
                                    <td><?php echo htmlspecialchars($jemaat['nama_pendeta']); ?></td>
                                    <td>
                                        <span class="badge bg-info"><?php echo $jemaat['jumlah_anggota']; ?></span>
                                    </td>
                                    <td>
                                        <span class="badge bg-success"><?php echo $jemaat['jumlah_peserta']; ?></span>
                                    </td>
                                    <td><?php echo htmlspecialchars($jemaat['nomor_telepon']); ?></td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-outline-primary" onclick="editJemaat(<?php echo htmlspecialchars(json_encode($jemaat)); ?>)" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <a href="peserta.php?jemaat_id=<?php echo $jemaat['id']; ?>" class="btn btn-outline-success" title="Lihat Peserta">
                                                <i class="fas fa-users"></i>
                                            </a>
                                            <button class="btn btn-outline-danger" onclick="deleteJemaat(<?php echo $jemaat['id']; ?>, '<?php echo htmlspecialchars($jemaat['nama_jemaat']); ?>')" title="Hapus">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <?php endwhile; ?>
                            <?php else: ?>
                                <tr>
                                    <td colspan="9" class="text-center py-4">
                                        <i class="fas fa-church fa-3x text-muted mb-3"></i>
                                        <h5 class="text-muted">Belum ada data jemaat</h5>
                                        <p class="text-muted">Mulai dengan menambahkan jemaat pertama</p>
                                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addJemaatModal">
                                            <i class="fas fa-plus me-2"></i>Tambah Jemaat Pertama
                                        </button>
                                    </td>
                                </tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Jemaat Modal -->
    <div class="modal fade" id="addJemaatModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-church me-2"></i>Tambah Data Jemaat Baru
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="add">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="nama_jemaat" class="form-label">Nama Jemaat <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="nama_jemaat" name="nama_jemaat" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="nama_pendeta" class="form-label">Nama Pendeta</label>
                                    <input type="text" class="form-control" id="nama_pendeta" name="nama_pendeta">
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="alamat_jemaat" class="form-label">Alamat Jemaat</label>
                            <textarea class="form-control" id="alamat_jemaat" name="alamat_jemaat" rows="3"></textarea>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="kota" class="form-label">Kota</label>
                                    <input type="text" class="form-control" id="kota" name="kota">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="provinsi" class="form-label">Provinsi</label>
                                    <input type="text" class="form-control" id="provinsi" name="provinsi">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="kode_pos" class="form-label">Kode Pos</label>
                                    <input type="text" class="form-control" id="kode_pos" name="kode_pos">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="nomor_telepon" class="form-label">Nomor Telepon</label>
                                    <input type="text" class="form-control" id="nomor_telepon" name="nomor_telepon">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="email" name="email">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="jumlah_anggota" class="form-label">Jumlah Anggota</label>
                                    <input type="number" class="form-control" id="jumlah_anggota" name="jumlah_anggota" value="0" min="0">
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="deskripsi" class="form-label">Deskripsi</label>
                            <textarea class="form-control" id="deskripsi" name="deskripsi" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Simpan
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Jemaat Modal -->
    <div class="modal fade" id="editJemaatModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit me-2"></i>Edit Data Jemaat
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="edit">
                        <input type="hidden" name="jemaat_id" id="edit_jemaat_id">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_nama_jemaat" class="form-label">Nama Jemaat <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="edit_nama_jemaat" name="nama_jemaat" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_nama_pendeta" class="form-label">Nama Pendeta</label>
                                    <input type="text" class="form-control" id="edit_nama_pendeta" name="nama_pendeta">
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="edit_alamat_jemaat" class="form-label">Alamat Jemaat</label>
                            <textarea class="form-control" id="edit_alamat_jemaat" name="alamat_jemaat" rows="3"></textarea>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="edit_kota" class="form-label">Kota</label>
                                    <input type="text" class="form-control" id="edit_kota" name="kota">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="edit_provinsi" class="form-label">Provinsi</label>
                                    <input type="text" class="form-control" id="edit_provinsi" name="provinsi">
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="edit_kode_pos" class="form-label">Kode Pos</label>
                                    <input type="text" class="form-control" id="edit_kode_pos" name="kode_pos">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_nomor_telepon" class="form-label">Nomor Telepon</label>
                                    <input type="text" class="form-control" id="edit_nomor_telepon" name="nomor_telepon">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="edit_email" name="email">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="edit_jumlah_anggota" class="form-label">Jumlah Anggota</label>
                                    <input type="number" class="form-control" id="edit_jumlah_anggota" name="jumlah_anggota" min="0">
                                </div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="edit_deskripsi" class="form-label">Deskripsi</label>
                            <textarea class="form-control" id="edit_deskripsi" name="deskripsi" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteJemaatModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-exclamation-triangle text-warning me-2"></i>Konfirmasi Hapus
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <p>Apakah Anda yakin ingin menghapus jemaat <strong id="delete_jemaat_nama"></strong>?</p>
                    <p class="text-danger">Tindakan ini tidak dapat dibatalkan!</p>
                </div>
                <form method="POST">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="jemaat_id" id="delete_jemaat_id">
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                        <button type="submit" class="btn btn-danger">
                            <i class="fas fa-trash me-2"></i>Hapus
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function editJemaat(jemaat) {
            document.getElementById('edit_jemaat_id').value = jemaat.id;
            document.getElementById('edit_nama_jemaat').value = jemaat.nama_jemaat;
            document.getElementById('edit_alamat_jemaat').value = jemaat.alamat_jemaat;
            document.getElementById('edit_kota').value = jemaat.kota;
            document.getElementById('edit_provinsi').value = jemaat.provinsi;
            document.getElementById('edit_kode_pos').value = jemaat.kode_pos;
            document.getElementById('edit_nomor_telepon').value = jemaat.nomor_telepon;
            document.getElementById('edit_email').value = jemaat.email;
            document.getElementById('edit_nama_pendeta').value = jemaat.nama_pendeta;
            document.getElementById('edit_jumlah_anggota').value = jemaat.jumlah_anggota;
            document.getElementById('edit_deskripsi').value = jemaat.deskripsi;
            
            new bootstrap.Modal(document.getElementById('editJemaatModal')).show();
        }
        
        function deleteJemaat(jemaatId, jemaatNama) {
            document.getElementById('delete_jemaat_id').value = jemaatId;
            document.getElementById('delete_jemaat_nama').textContent = jemaatNama;
            
            new bootstrap.Modal(document.getElementById('deleteJemaatModal')).show();
        }
    </script>
</body>
</html> 