<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Redirect jika belum login
if (!isLoggedIn()) {
    header("Location: index.php");
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    $user_id = $_SESSION['user_id'];
    $sql = "SELECT full_name FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $_SESSION['full_name'] = $user['full_name'];
    }
}

// Ambil data dokumen
$sql = "SELECT d.*, u.username as uploader_name FROM documents d 
        JOIN users u ON d.uploaded_by = u.id 
        ORDER BY d.created_at DESC";
$result = $conn->query($sql);

// Filter berdasarkan tipe file
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
if ($filter !== 'all') {
    $sql = "SELECT d.*, u.username as uploader_name FROM documents d 
            JOIN users u ON d.uploaded_by = u.id 
            WHERE d.file_type = ? 
            ORDER BY d.created_at DESC";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("s", $filter);
    $stmt->execute();
    $result = $stmt->get_result();
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Sistem Manajemen Dokumen</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/lightbox2@2.11.3/dist/css/lightbox.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Sistem Dokumen Persidangan
            </a>
            
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link active" href="dashboard.php">
                            <i class="fas fa-home me-1"></i>Dashboard
                        </a>
                    </li>
                    <?php if (canUpload()): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="upload.php">
                            <i class="fas fa-upload me-1"></i>Upload Dokumen
                        </a>
                    </li>
                    <?php endif; ?>
                    <li class="nav-item">
                        <a class="nav-link" href="jemaat.php">
                            <i class="fas fa-church me-1"></i>Data Jemaat
                        </a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="peserta.php">
                            <i class="fas fa-users me-1"></i>Data Peserta
                        </a>
                    </li>
                    <?php if (hasRole('administrator')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="users.php">
                            <i class="fas fa-users-cog me-1"></i>Kelola User
                        </a>
                    </li>
                    <?php endif; ?>
                    <?php if (hasRole('administrator') || hasRole('panitia')): ?>
                    <li class="nav-item">
                        <a class="nav-link" href="activity_logs.php">
                            <i class="fas fa-history me-1"></i>Log Aktivitas
                        </a>
                    </li>
                    <?php endif; ?>
                </ul>
                
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="profile.php">
                                <i class="fas fa-user-edit me-2"></i>Profile
                            </a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php">
                                <i class="fas fa-sign-out-alt me-2"></i>Logout
                            </a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <!-- Header -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h2><i class="fas fa-folder-open me-2"></i>Dokumen Persidangan</h2>
                <p class="text-muted">Kelola dan lihat semua dokumen persidangan</p>
            </div>
            <div class="col-md-4 text-end">
                <?php if (canUpload()): ?>
                <a href="upload.php" class="btn btn-primary">
                    <i class="fas fa-plus me-2"></i>Upload Dokumen
                </a>
                <?php endif; ?>
            </div>
        </div>

        <!-- Filter -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="btn-group" role="group">
                            <a href="?filter=all" class="btn btn-outline-primary <?php echo $filter === 'all' ? 'active' : ''; ?>">
                                <i class="fas fa-list me-1"></i>Semua
                            </a>
                            <a href="?filter=pdf" class="btn btn-outline-danger <?php echo $filter === 'pdf' ? 'active' : ''; ?>">
                                <i class="fas fa-file-pdf me-1"></i>PDF
                            </a>
                            <a href="?filter=image" class="btn btn-outline-success <?php echo $filter === 'image' ? 'active' : ''; ?>">
                                <i class="fas fa-file-image me-1"></i>Gambar
                            </a>
                            <a href="?filter=video" class="btn btn-outline-info <?php echo $filter === 'video' ? 'active' : ''; ?>">
                                <i class="fas fa-file-video me-1"></i>Video
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Documents Grid -->
        <div class="row">
            <?php if ($result->num_rows > 0): ?>
                <?php while ($doc = $result->fetch_assoc()): ?>
                <div class="col-md-4 col-lg-3 mb-4">
                    <div class="card h-100 document-card">
                        <div class="card-body text-center">
                            <?php if ($doc['file_type'] === 'pdf'): ?>
                                <!-- PDF Card -->
                                <div class="document-icon mb-3">
                                    <i class="fas fa-file-pdf fa-3x text-danger"></i>
                                </div>
                                <h6 class="card-title"><?php echo htmlspecialchars($doc['title']); ?></h6>
                                <p class="card-text small text-muted"><?php echo htmlspecialchars($doc['description']); ?></p>
                                <div class="mb-3">
                                    <span class="badge bg-danger">PDF</span>
                                    <small class="text-muted d-block"><?php echo formatFileSize($doc['file_size']); ?></small>
                                </div>
                                <a href="view_pdf.php?id=<?php echo $doc['id']; ?>" class="btn btn-outline-danger btn-sm">
                                    <i class="fas fa-book-open me-1"></i>Baca PDF
                                </a>
                                
                            <?php elseif ($doc['file_type'] === 'image'): ?>
                                <!-- Image Card -->
                                <div class="document-icon mb-3">
                                    <img src="uploads/<?php echo $doc['filename']; ?>" 
                                         class="img-thumbnail" 
                                         style="max-height: 100px; object-fit: cover;"
                                         alt="<?php echo htmlspecialchars($doc['title']); ?>">
                                </div>
                                <h6 class="card-title"><?php echo htmlspecialchars($doc['title']); ?></h6>
                                <p class="card-text small text-muted"><?php echo htmlspecialchars($doc['description']); ?></p>
                                <div class="mb-3">
                                    <span class="badge bg-success">Gambar</span>
                                    <small class="text-muted d-block"><?php echo formatFileSize($doc['file_size']); ?></small>
                                </div>
                                <a href="uploads/<?php echo $doc['filename']; ?>" 
                                   data-lightbox="gallery" 
                                   data-title="<?php echo htmlspecialchars($doc['title']); ?>"
                                   class="btn btn-outline-success btn-sm">
                                    <i class="fas fa-eye me-1"></i>Lihat Gambar
                                </a>
                                
                            <?php else: ?>
                                <!-- Video Card -->
                                <div class="document-icon mb-3">
                                    <i class="fas fa-file-video fa-3x text-primary"></i>
                                </div>
                                <h6 class="card-title"><?php echo htmlspecialchars($doc['title']); ?></h6>
                                <p class="card-text small text-muted"><?php echo htmlspecialchars($doc['description']); ?></p>
                                <div class="mb-3">
                                    <span class="badge bg-primary">Video</span>
                                    <small class="text-muted d-block"><?php echo formatFileSize($doc['file_size']); ?></small>
                                </div>
                                <a href="view_video.php?id=<?php echo $doc['id']; ?>" class="btn btn-outline-primary btn-sm">
                                    <i class="fas fa-play me-1"></i>Tonton Video
                                </a>
                            <?php endif; ?>
                        </div>
                        <div class="card-footer text-muted small">
                            <div class="d-flex justify-content-between">
                                <span><i class="fas fa-user me-1"></i><?php echo htmlspecialchars($doc['uploader_name']); ?></span>
                                <span><i class="fas fa-calendar me-1"></i><?php echo date('d/m/Y', strtotime($doc['created_at'])); ?></span>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endwhile; ?>
            <?php else: ?>
                <div class="col-12">
                    <div class="text-center py-5">
                        <i class="fas fa-folder-open fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">Belum ada dokumen</h4>
                        <p class="text-muted">Mulai dengan mengupload dokumen pertama Anda</p>
                        <?php if (canUpload()): ?>
                        <a href="upload.php" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>Upload Dokumen Pertama
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/lightbox2@2.11.3/dist/js/lightbox.min.js"></script>
</body>
</html> 