<?php
session_start();
require_once 'config/database.php';
require_once 'includes/functions.php';

// Cek apakah user sudah login
if (!isLoggedIn()) {
    header('Location: index.php');
    exit();
}

// Pastikan full_name tersedia di session
if (isset($_SESSION['user_id']) && !isset($_SESSION['full_name'])) {
    $user_id = $_SESSION['user_id'];
    $sql = "SELECT full_name FROM users WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows > 0) {
        $user = $result->fetch_assoc();
        $_SESSION['full_name'] = $user['full_name'];
    }
}

// Cek apakah user memiliki akses (admin atau panitia)
if (!hasRole('administrator') && !hasRole('panitia')) {
    header('Location: dashboard.php');
    exit();
}

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 20;
$offset = ($page - 1) * $limit;

// Filter
$filter_user_id = isset($_GET['user_id']) ? (int)$_GET['user_id'] : null;
$filter_action = isset($_GET['action']) ? $_GET['action'] : null;
$filter_table = isset($_GET['table_name']) ? $_GET['table_name'] : null;
$filter_date_from = isset($_GET['date_from']) ? $_GET['date_from'] : null;
$filter_date_to = isset($_GET['date_to']) ? $_GET['date_to'] : null;

// Ambil data log aktivitas
$logs = getActivityLogs($conn, $limit, $offset, $filter_user_id, $filter_action, $filter_table, $filter_date_from, $filter_date_to);
$total_logs = getTotalLogs($conn, $filter_user_id, $filter_action, $filter_table, $filter_date_from, $filter_date_to);
$total_pages = ceil($total_logs / $limit);

// Ambil daftar user untuk filter
$users_result = $conn->query("SELECT id, username, full_name FROM users ORDER BY username");
$users = [];
while ($row = $users_result->fetch_assoc()) {
    $users[] = $row;
}

// Ambil statistik aktivitas
$stats = getActivityStats($conn, $filter_user_id, $filter_date_from, $filter_date_to);
$stats_data = [];
while ($row = $stats->fetch_assoc()) {
    $stats_data[] = $row;
}
?>

<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Log Aktivitas - Sistem Persidangan</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="assets/css/style.css" rel="stylesheet">
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container">
            <a class="navbar-brand" href="dashboard.php">
                <img src="assets/images/logo-gereja-toraja.png" alt="Logo Gereja Toraja" style="width: 30px; height: 30px; margin-right: 10px;">
                Sistem Persidangan
            </a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav me-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="dashboard.php">Dashboard</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="jemaat.php">Jemaat</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="peserta.php">Peserta</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="upload.php">Upload Dokumen</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link active" href="activity_logs.php">Log Aktivitas</a>
                    </li>
                </ul>
                <ul class="navbar-nav">
                    <li class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user me-1"></i><?php echo htmlspecialchars($_SESSION['full_name'] ?? $_SESSION['username']); ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="logout.php">Logout</a></li>
                        </ul>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-history me-2"></i>Log Aktivitas</h2>
                    <div>
                        <button class="btn btn-outline-primary" onclick="exportLogs()">
                            <i class="fas fa-download me-1"></i>Export
                        </button>
                    </div>
                </div>

                <!-- Filter Form -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-filter me-2"></i>Filter Log</h5>
                    </div>
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-2">
                                <label for="user_id" class="form-label">User</label>
                                <select name="user_id" id="user_id" class="form-select">
                                    <option value="">Semua User</option>
                                    <?php foreach ($users as $user): ?>
                                    <option value="<?php echo $user['id']; ?>" <?php echo $filter_user_id == $user['id'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($user['username']); ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="action" class="form-label">Aksi</label>
                                <select name="action" id="action" class="form-select">
                                    <option value="">Semua Aksi</option>
                                    <option value="CREATE" <?php echo $filter_action == 'CREATE' ? 'selected' : ''; ?>>Create</option>
                                    <option value="UPDATE" <?php echo $filter_action == 'UPDATE' ? 'selected' : ''; ?>>Update</option>
                                    <option value="DELETE" <?php echo $filter_action == 'DELETE' ? 'selected' : ''; ?>>Delete</option>
                                    <option value="LOGIN" <?php echo $filter_action == 'LOGIN' ? 'selected' : ''; ?>>Login</option>
                                    <option value="LOGOUT" <?php echo $filter_action == 'LOGOUT' ? 'selected' : ''; ?>>Logout</option>
                                    <option value="UPLOAD" <?php echo $filter_action == 'UPLOAD' ? 'selected' : ''; ?>>Upload</option>
                                    <option value="DOWNLOAD" <?php echo $filter_action == 'DOWNLOAD' ? 'selected' : ''; ?>>Download</option>
                                    <option value="IMPORT" <?php echo $filter_action == 'IMPORT' ? 'selected' : ''; ?>>Import</option>
                                    <option value="EXPORT" <?php echo $filter_action == 'EXPORT' ? 'selected' : ''; ?>>Export</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="table_name" class="form-label">Tabel</label>
                                <select name="table_name" id="table_name" class="form-select">
                                    <option value="">Semua Tabel</option>
                                    <option value="users" <?php echo $filter_table == 'users' ? 'selected' : ''; ?>>Users</option>
                                    <option value="jemaat" <?php echo $filter_table == 'jemaat' ? 'selected' : ''; ?>>Jemaat</option>
                                    <option value="peserta" <?php echo $filter_table == 'peserta' ? 'selected' : ''; ?>>Peserta</option>
                                    <option value="documents" <?php echo $filter_table == 'documents' ? 'selected' : ''; ?>>Documents</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <label for="date_from" class="form-label">Dari Tanggal</label>
                                <input type="date" name="date_from" id="date_from" class="form-control" value="<?php echo $filter_date_from; ?>">
                            </div>
                            <div class="col-md-2">
                                <label for="date_to" class="form-label">Sampai Tanggal</label>
                                <input type="date" name="date_to" id="date_to" class="form-control" value="<?php echo $filter_date_to; ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">&nbsp;</label>
                                <div>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-search me-1"></i>Filter
                                    </button>
                                    <a href="activity_logs.php" class="btn btn-secondary">
                                        <i class="fas fa-times me-1"></i>Reset
                                    </a>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Statistik -->
                <?php if (!empty($stats_data)): ?>
                <div class="row mb-4">
                    <div class="col-12">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>Statistik Aktivitas</h5>
                            </div>
                            <div class="card-body">
                                <div class="row">
                                    <?php
                                    $action_counts = [];
                                    foreach ($stats_data as $stat) {
                                        if (!isset($action_counts[$stat['action']])) {
                                            $action_counts[$stat['action']] = 0;
                                        }
                                        $action_counts[$stat['action']] += $stat['count'];
                                    }
                                    ?>
                                    <?php foreach ($action_counts as $action => $count): ?>
                                    <div class="col-md-2 col-sm-4 col-6 mb-3">
                                        <div class="text-center">
                                            <div class="h4 text-<?php echo getActionBadgeColor($action); ?>">
                                                <i class="<?php echo getActionIcon($action); ?>"></i>
                                            </div>
                                            <div class="h5"><?php echo $count; ?></div>
                                            <small class="text-muted"><?php echo $action; ?></small>
                                        </div>
                                    </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Tabel Log -->
                <div class="card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">Daftar Aktivitas (<?php echo $total_logs; ?> total)</h5>
                            <small class="text-muted">Menampilkan <?php echo $limit; ?> data per halaman</small>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if ($logs->num_rows > 0): ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Waktu</th>
                                        <th>User</th>
                                        <th>Aksi</th>
                                        <th>Tabel</th>
                                        <th>Deskripsi</th>
                                        <th>IP Address</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while ($log = $logs->fetch_assoc()): ?>
                                    <tr>
                                        <td>
                                            <small class="text-muted">
                                                <?php echo formatActivityTime($log['created_at']); ?>
                                            </small>
                                            <br>
                                            <small><?php echo date('d/m/Y H:i:s', strtotime($log['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <div class="fw-bold"><?php echo htmlspecialchars($log['full_name']); ?></div>
                                            <small class="text-muted">@<?php echo htmlspecialchars($log['username']); ?></small>
                                            <br>
                                            <span class="badge bg-secondary"><?php echo ucfirst($log['role']); ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-<?php echo getActionBadgeColor($log['action']); ?>">
                                                <i class="<?php echo getActionIcon($log['action']); ?> me-1"></i>
                                                <?php echo $log['action']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($log['table_name']): ?>
                                            <span class="badge bg-info"><?php echo ucfirst($log['table_name']); ?></span>
                                            <?php if ($log['record_id']): ?>
                                            <br><small class="text-muted">ID: <?php echo $log['record_id']; ?></small>
                                            <?php endif; ?>
                                            <?php else: ?>
                                            <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($log['description']): ?>
                                            <div class="text-wrap" style="max-width: 300px;">
                                                <?php echo htmlspecialchars($log['description']); ?>
                                            </div>
                                            <?php else: ?>
                                            <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <small class="text-muted"><?php echo htmlspecialchars($log['ip_address']); ?></small>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($total_pages > 1): ?>
                        <nav aria-label="Page navigation" class="mt-4">
                            <ul class="pagination justify-content-center">
                                <?php if ($page > 1): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page - 1; ?>&user_id=<?php echo $filter_user_id; ?>&action=<?php echo $filter_action; ?>&table_name=<?php echo $filter_table; ?>&date_from=<?php echo $filter_date_from; ?>&date_to=<?php echo $filter_date_to; ?>">
                                        <i class="fas fa-chevron-left"></i>
                                    </a>
                                </li>
                                <?php endif; ?>

                                <?php
                                $start_page = max(1, $page - 2);
                                $end_page = min($total_pages, $page + 2);
                                
                                for ($i = $start_page; $i <= $end_page; $i++):
                                ?>
                                <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                    <a class="page-link" href="?page=<?php echo $i; ?>&user_id=<?php echo $filter_user_id; ?>&action=<?php echo $filter_action; ?>&table_name=<?php echo $filter_table; ?>&date_from=<?php echo $filter_date_from; ?>&date_to=<?php echo $filter_date_to; ?>">
                                        <?php echo $i; ?>
                                    </a>
                                </li>
                                <?php endfor; ?>

                                <?php if ($page < $total_pages): ?>
                                <li class="page-item">
                                    <a class="page-link" href="?page=<?php echo $page + 1; ?>&user_id=<?php echo $filter_user_id; ?>&action=<?php echo $filter_action; ?>&table_name=<?php echo $filter_table; ?>&date_from=<?php echo $filter_date_from; ?>&date_to=<?php echo $filter_date_to; ?>">
                                        <i class="fas fa-chevron-right"></i>
                                    </a>
                                </li>
                                <?php endif; ?>
                            </ul>
                        </nav>
                        <?php endif; ?>

                        <?php else: ?>
                        <div class="text-center py-5">
                            <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">Tidak ada log aktivitas</h5>
                            <p class="text-muted">Belum ada aktivitas yang tercatat dalam sistem.</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function exportLogs() {
            // Buat URL dengan parameter filter saat ini
            const urlParams = new URLSearchParams(window.location.search);
            const exportUrl = 'export_logs.php?' + urlParams.toString();
            window.open(exportUrl, '_blank');
        }
    </script>
</body>
</html> 